<?php

use Stripe\Stripe;
use Stripe\Charge;
use Stripe\PaymentIntent;
use Stripe\Exception\CardException;
use Stripe\Exception\RateLimitException;
use Stripe\Exception\InvalidRequestException;
use Stripe\Exception\AuthenticationException;
use Stripe\Exception\ApiConnectionException;
use Stripe\Exception\ApiErrorException;

class Tools_StripeRegular
{
    protected $privateKey;
    protected $publicKey;
    protected $paymentData;

    public function __construct($privateKey, array $paymentData)
    {
        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();

        if (empty($privateKey) && empty($stripepayOAuth)) {
            throw new Exception (print " Error, api key is empty");
        }
        $this->privateKey = $privateKey;
        $this->paymentData = $paymentData;
    }

    public function makeCharge()
    {
        $translator = Zend_Registry::get('Zend_Translate');
        Stripe::setApiKey($this->privateKey);
        $payResult = [];
        try {
            $charge = Charge::create(array(
                    'amount' => $this->paymentData['amount'] * 100,
                    'currency' => Models_Mapper_ShoppingConfig::getInstance()->getConfigParam('currency'),
                    'source' => $this->paymentData['stripeToken'],
                    'description' => '',
                    'metadata' => [
                        'Order Id' => $this->paymentData['cartId'],
                        'Customer name' => trim($this->paymentData['userFirstName'] . ' ' . $this->paymentData['userLastName']),
                        'Customer email' => $this->paymentData['userEmail'],
                    ],)
            );
        } catch (Exception $e) {
            $payResult['error'] = 1;
            $payResult['message'] = $e->getMessage();
            return $payResult;
        }
        if (!empty($charge) && $charge->status == 'succeeded') {
            try {
                $stripeTransactionMapper = Stripepay_Models_Mappers_StripepayTransactionMapper::getInstance();
                $stripeTransactionModel = new Stripepay_Models_Models_StripepayTransactionModel();

                $stripeTransactionModel->setPaymentType(Stripepay::PAYMENT_TYPE_CART);
                $stripeTransactionModel->setChargeId($charge->id);
                $stripeTransactionModel->setStripeCustomerId('');
                $stripeTransactionModel->setPaymentIntent('');
                $stripeTransactionModel->setStatus($charge->status);
                $stripeTransactionModel->setAmount($this->paymentData['amount']);
                $stripeTransactionModel->setCardType($charge->payment_method_details->card->brand);
                $stripeTransactionModel->setExpMonth($charge->payment_method_details->card->exp_month);
                $stripeTransactionModel->setExpYear($charge->payment_method_details->card->exp_year);
                $stripeTransactionModel->setCardFingerprint($charge->payment_method_details->card->fingerprint);
                $stripeTransactionModel->setCardMask($charge->payment_method_details->card->last4);
                $stripeTransactionModel->setToken($this->paymentData['stripeToken']);
                $stripeTransactionModel->setTotalTax($this->paymentData['totalTax']);
                $stripeTransactionModel->setCartId($this->paymentData['cartId']);
                $stripeTransactionModel->setOrderShippingAddress($this->paymentData['customerShipping']);
                $stripeTransactionModel->setOrderBillingAddress($this->paymentData['billingAddress']);
                $stripeTransactionModel->setOrderDateFull(Tools_System_Tools::convertDateFromTimezone('now'));
                $stripeTransactionModel->setOrderAmount($charge->amount);
                $stripeTransactionModel->setCurrency($charge->currency);
                $stripeTransactionModel->setTransactionType($charge->object);
                $stripeTransactionModel->setReceiptUrl($charge->receipt_url);
                $stripeTransactionModel->setPaymentMethod($charge->payment_method);
                $stripeTransactionModel->setFirstName($this->paymentData['userFirstName']);
                $stripeTransactionModel->setLastName($this->paymentData['userLastName']);
                $stripeTransactionModel->setEmail($this->paymentData['userEmail']);
                $stripeTransactionModel->setEmailSent('1');
                $stripeTransactionModel->setCustomerEmailSent('');

                $stripeTransactionMapper->save($stripeTransactionModel);

            } catch (Exception $e) {
                error_log($e->getMessage());
            }
            $payResult['error'] = 0;
            return $payResult;

        } else {
            $payResult['error'] = 1;
            $payResult['message'] = isset($charge->failure_message) ? $charge->failure_message : $translator->translate('Something went wrong');
            return $payResult;
        }
    }

    /**
     * Get domain name from url
     *
     * @param string $url url
     * @return string
     */
    public static function getDomainFromUrl($url)
    {
        $urlParts = parse_url($url);
        if (!empty($urlParts['host'])) {
            if (preg_match('/(?<domain>[a-z0-9\-]{1,63}\.[a-z\.]{2,6})$/i', $urlParts['host'], $matches)) {
                if (!empty($matches['domain'])) {
                    return $matches['domain'];
                }

            }
        }

        return '';

    }

    public static function parseOptionsString($optionsString) {
        $options = array();
        $parsed = array();
        parse_str($optionsString, $options);
        foreach($options as $keyString => $option) {
            $key          = preg_replace('~product-[0-9]*\-option\-([^0-9]*)*~', '$1', $keyString);
            $parsed[$key] = $option;
        }
        return $parsed;
    }

    /**
     * Get currency info
     *
     * @param bool $jsFormat format locale flag
     * @return array
     */
    public static function getCurrencyInfo($jsFormat = true)
    {
        $shoppingConfig = Models_Mapper_ShoppingConfig::getInstance()->getConfigParams();

        $country = $shoppingConfig['country'];
        if (!empty($shoppingConfig['currencyCountry'])) {
            $country = $shoppingConfig['currencyCountry'];
        }

        $locale = Zend_Locale::getLocaleToTerritory($country);
        $currency = $shoppingConfig['currency'];
        if ($jsFormat === true) {
            $locale = str_replace('_', '-', $locale);
        }

        return array('locale' => $locale, 'currency' => $currency);
    }

    /**
     * Get general stripe config param
     *
     * @param string $paramName
     * @return array|mixed
     */
    public static function getStripeConfig($paramName = '')
    {
        $stripepayConfigMapper = Stripepay_Models_Mappers_StripepayConfigMapper::getInstance();
        $stripeConfig = $stripepayConfigMapper->getConfigParams();

        $params = array(
            'privateKey' => $stripeConfig['useSandBox'] == "0" ? $stripeConfig['privateKey'] : $stripeConfig['privateKeyTest'],
            'publishKey' => $stripeConfig['useSandBox'] == "0" ? $stripeConfig['publishableKey'] : $stripeConfig['publishableKeyTest'],
            'useSandBox' => isset($stripeConfig['useSandBox']) ? $stripeConfig['useSandBox'] == "1" : true,
            'disableLink' => isset($stripeConfig['disableLink']) ? $stripeConfig['disableLink'] == "1" : true,
            'endpointSecret' => isset($stripeConfig['endpointSecret']) ? $stripeConfig['endpointSecret'] : ''
        );

        if (!empty($paramName) && isset($params[$paramName])) {
            return $params[$paramName];
        }

        return $params;
    }

    /**
     * @param string $stripePrivateKey stripe payment key
     * @return string
     */
    public static function generatePaymentIntent($stripePrivateKey = '')
    {
        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();

        if (empty($stripePrivateKey) && empty($stripepayOAuth)) {
            $stripePrivateKey = self::getStripeConfig('privateKey');
        }
        $intent = Tools_RecurringPaymentStripepay::createSetupIntent($stripePrivateKey);

        if(!empty($intent['intent'])) {
            $intentData = json_decode($intent['intent'], true);

            if(!empty($intentData)) {
                $intent = (object) $intentData;
            }
        }

        $stripeSecretKey = '';
        if (!is_null($intent) && is_object($intent)) {
            $stripeSecretKey = $intent->client_secret;
        }

        return $stripeSecretKey;
    }

    public static function saveIntentTransactionLog($paymentIntent, $paymentData)
    {
        try {
            $stripeTransactionMapper = Stripepay_Models_Mappers_StripepayTransactionMapper::getInstance();
            $stripeTransactionModel = new Stripepay_Models_Models_StripepayTransactionModel();

            $stripeTransactionModel->setPaymentType($paymentIntent->payment_method_types[0]);

            $charges = isset($paymentIntent->charges->data) ? $paymentIntent->charges->data : '';
            if(!empty($charges) && is_array($charges)) {
                $charges = end($charges);
            }

            $chargeId = isset($charges->id) ? $charges->id : '';
            $stripeTransactionModel->setChargeId($chargeId);
            $stripeTransactionModel->setStripeCustomerId('');
            $stripeTransactionModel->setPaymentIntent($paymentIntent->id);
            $stripeTransactionModel->setStatus($paymentIntent->status);
            $stripeTransactionModel->setAmount($paymentIntent->amount);

            $paymentMethodDetails = isset($charges->payment_method_details) ? $charges->payment_method_details : '';

            $brand = '';
            $expMonth = '';
            $expYear = '';
            $fingerprint = '';
            $last4 = '';
            if(!empty($paymentMethodDetails)) {
                $brand = isset($paymentMethodDetails->card->brand) ? $paymentMethodDetails->card->brand : '';
                $expMonth = isset($paymentMethodDetails->card->exp_month) ? $paymentMethodDetails->card->exp_month : '';
                $expYear = isset($paymentMethodDetails->card->exp_year) ? $paymentMethodDetails->card->exp_year : '';
                $fingerprint = isset($paymentMethodDetails->card->fingerprint) ? $paymentMethodDetails->card->fingerprint : '';
                $last4 = isset($paymentMethodDetails->card->last4) ? $paymentMethodDetails->card->last4 : '';
            }

            $stripeTransactionModel->setCardType($brand);
            $stripeTransactionModel->setExpMonth($expMonth);
            $stripeTransactionModel->setExpYear($expYear);
            $stripeTransactionModel->setCardFingerprint($fingerprint);
            $stripeTransactionModel->setCardMask($last4);
            $stripeTransactionModel->setToken(isset($paymentData['stripeToken']) ? $paymentData['stripeToken'] : '');
            $stripeTransactionModel->setTotalTax($paymentData['totalTax']);
            $stripeTransactionModel->setCartId($paymentData['cartId']);
            $stripeTransactionModel->setOrderShippingAddress($paymentData['customerShipping']);
            $stripeTransactionModel->setOrderBillingAddress($paymentData['billingAddress']);
            $stripeTransactionModel->setOrderDateFull(Tools_System_Tools::convertDateFromTimezone('now'));
            $stripeTransactionModel->setOrderAmount($paymentIntent->amount);
            $stripeTransactionModel->setCurrency($paymentIntent->currency);

            $type = isset($charges->object) ? $charges->object : '';
            $stripeTransactionModel->setTransactionType($type);

            $receiptUrl = isset($charges->receipt_url) ? $charges->receipt_url : '';
            $stripeTransactionModel->setReceiptUrl($receiptUrl);

            $paymentMethodId = isset($charges->payment_method) ? $charges->payment_method : '';
            $stripeTransactionModel->setPaymentMethod($paymentMethodId);
            $stripeTransactionModel->setIntentClientSecret($paymentIntent->client_secret);
            $stripeTransactionModel->setFirstName($paymentData['userFirstName']);
            $stripeTransactionModel->setLastName($paymentData['userLastName']);
            $stripeTransactionModel->setEmail($paymentData['userEmail']);
            $stripeTransactionModel->setEmailSent('1');
            $stripeTransactionModel->setCustomerEmailSent('');

            $stripeTransactionMapper->save($stripeTransactionModel);
        } catch (Exception $e) {
            error_log($e->getMessage());
        }
    }



}
