<?php

use Stripe\PaymentMethod;
use Stripe\PaymentIntent;
use Stripe\Charge;

class Tools_StripeRecurrent
{
    protected $privateKey;
    protected $publicKey;
    protected $paymentData;

    const PAYMENT_INTENT_STATUS_ERROR_REQUIRES_PAYMENT_METHOD = 'requires_payment_method';
    const PAYMENT_INTENT_STATUS_ERROR_AUTHENTICATION_REQUIRED = 'authentication_required';
    const PAYMENT_INTENT_STATUS_SUCCESS = 'succeeded';
    const PAYMENT_INTENT_STATUS_ERROR = 'error';

    public function __construct($privateKey, $paymentMethod, array $paymentData)
    {
        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();

        if (empty($privateKey) && empty($stripepayOAuth)) {
            throw new Exception (print " Error, api key is empty");
        }
        $this->privateKey = $privateKey;
        $this->paymentMethod = $paymentMethod;
        $this->paymentData = $paymentData;
    }

    public function processPaymentIntent($paymentType, $customerId = null)
    {
        $payResult = [];
        if (!$customerId) {
            $customerId = Tools_RecurringPaymentStripepay::saveCustomerCard($this->privateKey, $this->paymentMethod);
        }
        if ($customerId) {
            $currency = Models_Mapper_ShoppingConfig::getInstance()->getConfigParam('currency');
            $paymentMethod = Tools_RecurringPaymentStripepay::getCustomerPaymentMethods($this->privateKey, $customerId)->data[0];
            if (!is_null($paymentMethod)) {
                $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();
                if(!empty($stripepayOAuth)) {
                    $paymentMethod = (object) $paymentMethod;
                }
                $metaData = [
                    'Order Id' => $this->paymentData['cartId'],
                    'Customer name' => trim($this->paymentData['userFirstName'] . ' ' . $this->paymentData['userLastName']),
                    'Customer email' => $this->paymentData['userEmail'],
                ];
                $paymentResult = Tools_RecurringPaymentStripepay::chargeCustomerCard($this->privateKey, $customerId, $paymentMethod->id, $this->paymentData['amount'], $currency, $metaData);
                if (isset($paymentResult['paymentIntent']) && !is_null($paymentResult['paymentIntent'])) {
                    if ($paymentResult['paymentIntent']->status == self::PAYMENT_INTENT_STATUS_SUCCESS) {
                        $status = self::PAYMENT_INTENT_STATUS_SUCCESS;
                        $payResult['intentId'] = $paymentResult['paymentIntent']->id;
                        $payResult['paymentMethodId'] = $paymentMethod->id;
                    } elseif (isset($paymentResult['errorCode'])) {
                        $status = $paymentResult['errorCode'] ? $paymentResult['errorCode'] : self::PAYMENT_INTENT_STATUS_ERROR;
                    } else {
                        $status = self::PAYMENT_INTENT_STATUS_ERROR;
                    }
                    $this->_saveTransaction($paymentMethod, $paymentResult['paymentIntent'], $customerId, $status, $paymentType);
                    $payResult['error'] = $status == self::PAYMENT_INTENT_STATUS_SUCCESS ? 0 : 1;
                    if ($payResult['error'] == 1) {
                        $payResult['message'] = isset($paymentResult['paymentIntent']->last_payment_error->message) ? $paymentResult['paymentIntent']->last_payment_error->message : Stripepay::ERROR_PAYMENT_COMMON;
                    }
                    return $payResult;
                } else {
                    $payResult['error'] = 1;
                    $payResult['message'] = isset($paymentResult['errorCode']) ? $paymentResult['errorCode'] : Stripepay::ERROR_PAYMENT_COMMON;
                }
            } else {
                $payResult['error'] = 1;
                $payResult['message'] = Stripepay::ERROR_PAYMENT_METHOD_NOT_FOUND;
            }
        } else {
            $payResult['error'] = 1;
            $payResult['message'] = Stripepay::ERROR_CUSTOMER_NOT_FOUND;
        }

        return $payResult;
    }

    protected function _saveTransaction($paymentMethod, $paymentIntent, $customerId, $status, $paymentType)
    {
        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();
        if(!empty($stripepayOAuth)) {
            $charge = $paymentIntent->latest_charge;
            $cardBrand = $paymentMethod->card['brand'];
            $cardExpMonth = $paymentMethod->card['exp_month'];
            $cardExpYear = $paymentMethod->card['exp_year'];
            $cardFingerprint = $paymentMethod->card['fingerprint'];
            $cardLast4 = $paymentMethod->card['last4'];
        } else {
            $charge = is_array($paymentIntent->charges->data) ? array_pop($paymentIntent->charges->data) : null;
            $cardBrand = $paymentMethod->card->brand;
            $cardExpMonth = $paymentMethod->card->exp_month;
            $cardExpYear = $paymentMethod->card->exp_year;
            $cardFingerprint = $paymentMethod->card->fingerprint;
            $cardLast4 = $paymentMethod->card->last4;
        }

        try {
            $stripeTransactionMapper = Stripepay_Models_Mappers_StripepayTransactionMapper::getInstance();
            $stripeTransactionModel = new Stripepay_Models_Models_StripepayTransactionModel();

            $stripeTransactionModel->setPaymentType($paymentType);
            if(!empty($stripepayOAuth)) {
                $stripeTransactionModel->setChargeId($charge);
            } else {
                $stripeTransactionModel->setChargeId($charge instanceof Charge ? $charge->id : '');
            }

            $stripeTransactionModel->setStripeCustomerId($customerId);
            $stripeTransactionModel->setPaymentIntent($paymentIntent->id);
            $stripeTransactionModel->setStatus($status);
            $stripeTransactionModel->setAmount($this->paymentData['amount']);
            $stripeTransactionModel->setCardType($cardBrand);
            $stripeTransactionModel->setExpMonth($cardExpMonth);
            $stripeTransactionModel->setExpYear($cardExpYear);
            $stripeTransactionModel->setCardFingerprint($cardFingerprint);
            $stripeTransactionModel->setCardMask($cardLast4);
            $stripeTransactionModel->setToken(isset($this->paymentData['stripeToken']) ? $this->paymentData['stripeToken'] : '');
            $stripeTransactionModel->setTotalTax($this->paymentData['totalTax']);
            $stripeTransactionModel->setCartId($this->paymentData['cartId']);
            $stripeTransactionModel->setOrderShippingAddress($this->paymentData['customerShipping']);
            $stripeTransactionModel->setOrderBillingAddress($this->paymentData['billingAddress']);
            $stripeTransactionModel->setOrderDateFull(Tools_System_Tools::convertDateFromTimezone('now'));
            $stripeTransactionModel->setOrderAmount($paymentIntent->amount);
            $stripeTransactionModel->setCurrency($paymentIntent->currency);
            $stripeTransactionModel->setTransactionType($paymentMethod->type);
            $stripeTransactionModel->setReceiptUrl('');
            $stripeTransactionModel->setPaymentMethod($paymentMethod->id);
            $stripeTransactionModel->setIntentClientSecret($paymentIntent->client_secret);
            $stripeTransactionModel->setFirstName($this->paymentData['userFirstName']);
            $stripeTransactionModel->setLastName($this->paymentData['userLastName']);
            $stripeTransactionModel->setEmail($this->paymentData['userEmail']);
            $stripeTransactionModel->setEmailSent('1');
            $stripeTransactionModel->setCustomerEmailSent('');

            $stripeTransactionMapper->save($stripeTransactionModel);

        } catch (Exception $e) {
            error_log($e->getMessage());
        }
    }

}