<?php

use Stripe\Stripe;
use Stripe\Customer;
use Stripe\SetupIntent;
use Stripe\PaymentIntent;
use Stripe\PaymentMethod;
use Stripe\Exception\CardException;
use Stripe\Exception\RateLimitException;
use Stripe\Exception\InvalidRequestException;
use Stripe\Exception\AuthenticationException;
use Stripe\Exception\ApiConnectionException;
use Stripe\Exception\ApiErrorException;
use Stripe\Collection;
use Stripe\Refund;

class Tools_RecurringPaymentStripepay implements Interfaces_RecurringPayment
{
    protected $_recurringPaymentData;
    protected $_translator;

    public function __construct($data = array())
    {
        $this->_translator = Zend_Registry::get('Zend_Translate');
        $this->_recurringPaymentData = $data;
    }

    public function updateRecurringPayment()
    {
        $recurrentCartId = isset($this->_recurringPaymentData['cartId']) ? $this->_recurringPaymentData['cartId'] : null;
        if (empty($recurrentCartId)) {
            return array('error' => true, 'message' => $this->_translator->translate('Missing cartId'));
        }
        if (isset($this->_recurringPaymentData['nextBillingDate'])) {
            return array('error' => true, 'message' => $this->_translator->translate('Billing date cannot be modified'));
        }
        $recurringPaymentsMapper = Store_Mapper_RecurringPaymentsMapper::getInstance();
        $recurringPayment = $recurringPaymentsMapper->getByCartId($recurrentCartId);
        if (!empty($recurringPayment) && $recurringPayment[0] instanceof Store_Model_RecurringPayments) {
            $recurringPayment = $recurringPayment[0];
            if ($recurringPayment->getRecurringStatus() == Store_Model_RecurringPayments::CANCELED_RECURRING_PAYMENT) {
                return array('error' => true, 'message' => $this->_translator->translate('The subscription cannot be reactivated'));
            }
            $paymentCycle = strtolower($this->_recurringPaymentData['paymentCycle']);
            if(!empty($paymentCycle)) {
                if($paymentCycle == 'day' || $paymentCycle == 'week') {
                    return array('error' => true, 'message' => $this->_translator->translate('Stripepay does not allow use payment period' . ' "'. $this->_recurringPaymentData['paymentCycle']. '"'));
                }

                $period = '';
                if($paymentCycle == 'month') {
                    $period = '+1 month';
                } elseif ($paymentCycle == 'month-two') {
                    $period = '+2 month';
                } elseif ($paymentCycle == 'quarter') {
                    $period = '+3 month';
                } elseif ($paymentCycle == 'semester') {
                    $period = '+6 month';
                } elseif ($paymentCycle == 'year') {
                    $period = '+1 year';
                }

                if(!empty($period)) {
                    $nextPaymentDate = date('Y-m-d', strtotime($period));
                    /*$nextPaymentDate = date('Y-m-d', strtotime('2023-01-30' . $period));*/

                    $recurringPayment->setNextPaymentDate($nextPaymentDate);
                    $recurringPayment->setPaymentPeriod(strtoupper($period));
                }
            }

            $recurringPayment->setRecurringStatus(Store_Model_RecurringPayments::ACTIVE_RECURRING_PAYMENT);
            $recurringPaymentsMapper->save($recurringPayment);
            $recurrentCart = Models_Mapper_CartSessionMapper::getInstance()->find($recurrentCartId);
            $user = Application_Model_Mappers_UserMapper::getInstance()->find($recurrentCart->getUserId());

            $recurringPayment->registerObserver(new Tools_Mail_Watchdog(array(
                'trigger' => Tools_StripepayMailWatchdog::TRIGGER_SUBSCRIPTION_RECURRING_REACTIVATED,
                'fullName' => $user->getFullName(),
                'userEmail' => $user->getEmail(),
                'userId' => $recurrentCart->getUserId()
            )));
            $recurringPayment->notifyObservers();
        } else {
            return array('error' => true, 'message' => $this->_translator->translate('Subscription not found'));
        }

        return array('error' => false, 'message' => $this->_translator->translate('Subscription reactivated'));
    }

    public function suspendRecurringPayment()
    {
        $recurrentCartId = isset($this->_recurringPaymentData['cartId']) ? $this->_recurringPaymentData['cartId'] : null;
        if (empty($recurrentCartId)) {
            return array('error' => true, 'message' => $this->_translator->translate('Missing cartId'));
        }
        $recurringPaymentsMapper = Store_Mapper_RecurringPaymentsMapper::getInstance();
        $recurringPayment = $recurringPaymentsMapper->getByCartId($recurrentCartId);
        if (!empty($recurringPayment) && $recurringPayment[0] instanceof Store_Model_RecurringPayments) {
            $recurringPayment = $recurringPayment[0];
            if ($recurringPayment->getRecurringStatus() == Store_Model_RecurringPayments::CANCELED_RECURRING_PAYMENT) {
                return array('error' => true, 'message' => $this->_translator->translate('The subscription cannot be suspended'));
            }
            $recurringPayment->setRecurringStatus(Store_Model_RecurringPayments::SUSPENDED_RECURRING_PAYMENT);
            $recurringPaymentsMapper->save($recurringPayment);
            $recurrentCart = Models_Mapper_CartSessionMapper::getInstance()->find($recurrentCartId);
            $user = Application_Model_Mappers_UserMapper::getInstance()->find($recurrentCart->getUserId());

            $recurringPayment->registerObserver(new Tools_Mail_Watchdog(array(
                'trigger' => Tools_StripepayMailWatchdog::TRIGGER_SUBSCRIPTION_RECURRING_SUSPENDED,
                'fullName' => $user->getFullName(),
                'userEmail' => $user->getEmail(),
                'userId' => $recurrentCart->getUserId()
            )));
            $recurringPayment->notifyObservers();
        } else {
            return array('error' => true, 'message' => $this->_translator->translate('Subscription not found'));
        }

        return array('error' => false, 'message' => $this->_translator->translate('Subscription suspended'));
    }

    public function cancelRecurringPayment()
    {
        $recurrentCartId = isset($this->_recurringPaymentData['cartId']) ? $this->_recurringPaymentData['cartId'] : null;
        if (empty($recurrentCartId)) {
            return array('error' => true, 'message' => $this->_translator->translate('Missing cartId'));
        }
        $recurringPaymentsMapper = Store_Mapper_RecurringPaymentsMapper::getInstance();
        $recurringPayment = $recurringPaymentsMapper->getByCartId($recurrentCartId);
        if (!empty($recurringPayment) && $recurringPayment[0] instanceof Store_Model_RecurringPayments) {
            $recurringPayment = $recurringPayment[0];
            if ($recurringPayment->getRecurringStatus() == Store_Model_RecurringPayments::CANCELED_RECURRING_PAYMENT) {
                return array('error' => true, 'message' => $this->_translator->translate('The subscription already cancelled'));
            }
            $recurringPayment->setRecurringStatus(Store_Model_RecurringPayments::CANCELED_RECURRING_PAYMENT);
            $recurringPaymentsMapper->save($recurringPayment);
            $recurrentCart = Models_Mapper_CartSessionMapper::getInstance()->find($recurrentCartId);
            $user = Application_Model_Mappers_UserMapper::getInstance()->find($recurrentCart->getUserId());

            $recurringPayment->registerObserver(new Tools_Mail_Watchdog(array(
                'trigger' => Tools_StripepayMailWatchdog::TRIGGER_SUBSCRIPTION_RECURRING_CANCELED,
                'fullName' => $user->getFullName(),
                'userEmail' => $user->getEmail(),
                'userId' => $recurrentCart->getUserId()
            )));
            $recurringPayment->notifyObservers();
        } else {
            return array('error' => true, 'message' => $this->_translator->translate('Subscription not found'));
        }

        return array('error' => false, 'message' => $this->_translator->translate('Subscription canceled'));
    }

    /**
     * A SetupIntent is an object that represents your intent to set up a customer’s card for future payments.
     * https://stripe.com/docs/payments/save-and-reuse#web-create-setup-intent
     * The SetupIntent object contains a client secret, a unique key that you need to pass to Stripe.js on the client
     * side to collect card details. The client secret lets you perform certain actions on the client, such as
     * confirming the setup and updating payment method details, while hiding sensitive information like $customer.
     * The client secret can be used to validate and authenticate card details via the credit card networks.
     * Because of its sensitive nature, the client secret should not be logged, embedded in URLs, or exposed to anyone
     * other than the customer.
     * @param string $apiSecretKey - api secret key.
     * @throws
     */
    public static function createSetupIntent($apiSecretKey = '')
    {
        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();

        if(!empty($stripepayOAuth)) {
            return Tools_StripeOAuthTools::createSetupIntentMojo();
        } else {
            Stripe::setApiKey($apiSecretKey);
            return SetupIntent::create();
        }
    }

    /**
     * Once the SetupIntent has succeeded, associate the card details with a Customer
     * When creating a new Customer, pass a PaymentMethod ID to immediately add a payment method.
     * @param string $apiSecretKey - api secret key.
     * @param string $paymentMethodId
     * @return string $customerId
     * @throws
     */
    public static function saveCustomerCard($apiSecretKey, $paymentMethodId, $customerId = null)
    {
        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();

        if ($customerId) {
            if(!empty($stripepayOAuth)) {
                // return Obj
                $responcePaymentMethod = Tools_StripeOAuthTools::retrievePaymentMethodMojo($paymentMethodId);

                if(!empty($responcePaymentMethod)) {
                    $responcePaymentMethodData = json_decode($responcePaymentMethod, true);

                    if(!empty($responcePaymentMethodData)) {
                        $payment_method = (object) $responcePaymentMethodData;
                    }
                }
            } else {
                Stripe::setApiKey($apiSecretKey);
                $payment_method = PaymentMethod::retrieve($paymentMethodId);
            }

            $payment_method->attach(['customer' => $customerId]);
            return $customerId;
        } else {
            if(!empty($stripepayOAuth)) {
                // return Customer Obj
                $customer = Tools_StripeOAuthTools::createCustomerMojo($paymentMethodId);

                if(!empty($customer['customer'])) {
                    $customerData = json_decode($customer['customer'], true);

                    if(!empty($customerData)) {
                        $customer = (object) $customerData;

                        return $customer->id;
                    }
                }

                return null;
            } else {
                Stripe::setApiKey($apiSecretKey);
                // This creates a new Customer and attaches the PaymentMethod in one API call.
                $customer = Customer::create([
                    'payment_method' => $paymentMethodId,
                ]);

                return ($customer instanceof Customer) ? $customer->id : null;
            }
        }
    }

    /**
     * Creates a PaymentIntent with the amount and currency of the payment.
     *
     * @param string $apiSecretKey - api secret key.
     * @param string $customerId
     * @param string $paymentMethodId
     * @param int $amount
     * @param string $currency
     * @param array $metaData
     * @return array
     * @throws
     */
    public static function chargeCustomerCard($apiSecretKey, $customerId, $paymentMethodId, $amount, $currency = 'usd', $metaData = [])
    {
        $result = [];
        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();
        try {
            $dataRequest = array(
                'amount' => $amount * 100,
                'currency' => strtolower($currency),
                'customer' => $customerId,
                'payment_method' => $paymentMethodId,
                'off_session' => true,
                'confirm' => true,
                'metadata' => $metaData
            );

            if(!empty($stripepayOAuth)) {
                $configData = Application_Model_Mappers_ConfigMapper::getInstance()->getConfig();
                $websiteId = $configData['websiteId'];
                $dataRequest['metadata']['websiteId'] = $websiteId;

                //return array with PaymentIntent obj
                $responcePaymentIntent = Tools_StripeOAuthTools::createPaymentIntentMojo($dataRequest);

                if(!empty($responcePaymentIntent)) {
                    $responcePaymentIntentData = json_decode($responcePaymentIntent, true);

                    if(!empty($responcePaymentIntentData)) {
                        $result['paymentIntent'] = (object) $responcePaymentIntentData;
                    }
                }
            } else {
                Stripe::setApiKey($apiSecretKey);
                $result['paymentIntent'] = PaymentIntent::create($dataRequest);
            }
        } catch (CardException $e) {
            // Error code will be authentication_required if authentication is needed
            $payment_intent_id = $e->getError()->payment_intent->id;
            $result['errorCode'] = $e->getError()->code;

            if(!empty($stripepayOAuth)) {
                //return Obj
                $responcePaymentIntent = Tools_StripeOAuthTools::retrievePaymentIntentMojo($payment_intent_id);
                if(!empty($responcePaymentIntent)) {
                    $responcePaymentIntentData = json_decode($responcePaymentIntent, true);

                    if(!empty($responcePaymentIntentData)) {
                        $result['paymentIntent'] = (object) $responcePaymentIntentData;
                    }
                }

            } else {
                $result['paymentIntent'] = PaymentIntent::retrieve($payment_intent_id);
            }
        } catch (RateLimitException $e) {
            // Too many requests made to the API too quickly
            $result['errorCode'] = $e->getError()->code;
        } catch (InvalidRequestException $e) {
            // Invalid parameters were supplied to Stripe's API
            $result['errorCode'] = $e->getError()->code;
        } catch (AuthenticationException $e) {
            // Authentication with Stripe's API failed
            // (maybe you changed API keys recently)
            $result['errorCode'] = $e->getError()->code;
        } catch (ApiConnectionException $e) {
            // Network communication with Stripe failed
            $result['errorCode'] = $e->getError()->code;
        } catch (ApiErrorException $e) {
            // Display a very generic error to the user, and maybe send
            // yourself an email
            $result['errorCode'] = $e->getError()->code;
        } catch (Exception $e) {
            // Something else happened, completely unrelated to Stripe
            $result['errorCode'] = $e->getError()->code;
        }

        return $result;
    }

    /**
     * Returns a list of PaymentMethods for a given Customer
     * @param string $apiSecretKey - api secret key.
     * @param string $customerId
     * @throws
     */
    public static function getCustomerPaymentMethods($apiSecretKey, $customerId)
    {
        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();

        if(!empty($stripepayOAuth)) {
            try {
                // return PaymentMethod Obj
                $result = Tools_StripeOAuthTools::allPaymentMethodMojo($customerId, 'card');

                if(!empty($result['paymentMethod'])) {
                    $resultData = json_decode($result['paymentMethod'], true);

                    if(!empty($resultData)) {
                        $resultData = (object) $resultData;
                        return $resultData;
                    }
                }
                return null;
            } catch (Exception $e) {
            }
        } else {
            Stripe::setApiKey($apiSecretKey);
            try {
                $result = PaymentMethod::all([
                    'customer' => $customerId,
                    'type' => 'card',
                ]);
            } catch (Exception $e) {
            }

            return isset($result) ? $result : null;
        }
    }

    /**
     * Customer objects allow to perform recurring charges, and to track multiple charges, that are associated with
     * the same customer.
     *
     * @param string $apiSecretKey - api secret key.
     * @param string $customerName - The customer’s full name or business name.
     * @param string(512) $customerEmail - Customer’s email address. It’s displayed alongside the customer in your dashboard and can be useful for searching and tracking. This may be up to 512 characters.
     * @return Customer
     * @throws
     */
    public static function createCustomer($apiSecretKey, $customerName = '', $customerEmail = '')
    {
        Stripe::setApiKey($apiSecretKey);

        return Customer::create([
            'name' => $customerName,
            'email' => $customerEmail,
        ]);
    }

    /**
     * Updates the specified customer by setting the values of the parameters passed. Any parameters not provided will
     * be left unchanged.
     * https://stripe.com/docs/api/customers/update
     *
     * @param string $apiSecretKey - api secret key
     * @param string $customerId - The identifier of the customer to be updated.
     * @param array $customerData
     * @return Customer - the customer object if the update succeeded. Throws an error if update parameters
     * are invalid (e.g. specifying an invalid coupon or an invalid source).
     * @throws
     */
    public static function updateCustomer($apiSecretKey, $customerId, $customerData = [])
    {
        Stripe::setApiKey($apiSecretKey);

        return Customer::update(
            $customerId,
            ['metadata' => $customerData]
        );
    }

    /**
     * Permanently deletes a customer. It cannot be undone. Also immediately cancels any active subscriptions on the customer.
     * https://stripe.com/docs/api/customers/delete
     *
     * @param string $apiSecretKey - api secret key
     * @param string $customerId - The identifier of the customer to be deleted.
     * @return object - an object with a deleted parameter on success. If the customer ID does not exist, this call throws an error.
     * @throws
     */
    public static function deleteCustomer($apiSecretKey, $customerId)
    {
        Stripe::setApiKey($apiSecretKey, $customerId);

        $customer = Customer::retrieve(
            $customerId
        );
        return $customer->delete();
    }

    /**
     * Retrieves the details of an existing customer. You need only supply the unique customer identifier that
     * was returned upon customer creation.
     * https://stripe.com/docs/api/customers/retrieve
     *
     * @param string $apiSecretKey - api secret key.
     * @param string $customerId - The identifier of the customer to be retrieved.
     * @return Customer - a customer object if a valid identifier was provided.
     * When requesting the ID of a customer that has been deleted, a subset of the customer’s information will be
     * returned, including a deleted property, which will be true.
     * @throws
     */
    public static function retrieveCustomer($apiSecretKey, $customerId)
    {
        Stripe::setApiKey($apiSecretKey);

        return Customer::retrieve($customerId);
    }

    /**
     * Returns a list of your customers. The customers are returned sorted by creation date, with the most recent customers appearing first.
     * https://stripe.com/docs/api/customers/list
     *
     * @param string $apiSecretKey - api secret key.
     * @param int $limit
     * @return object
     * - an associative array with a "data" property that contains an array of up to $limit customers,
     * starting after customer starting_after. Passing an optional email will result in filtering to customers with
     * only that exact email address. Each entry in the array is a separate customer object. If no more customers are
     * available, the resulting array will be empty. This request should never throw an error.
     * @throws
     */
    public static function listAllCustomers($apiSecretKey, $limit = 5)
    {
        Stripe::setApiKey($apiSecretKey);

        return Customer::all(['limit' => $limit]);
    }

    public static function refundPaymentIntent($apiSecretKey, $paymentIntent, $amount = false)
    {
        $result = [];
        $translator = Zend_Registry::get('Zend_Translate');
        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();

        if(!empty($stripepayOAuth)) {
            try {
                if($amount) {
                    $requestData = array(
                        'amount' => $amount,
                        'payment_intent' => $paymentIntent
                    );
                    //return Refund Obj
                    $refundResponce = Tools_StripeOAuthTools::createRefundMojo($requestData);

                    if(!empty($refundResponce)) {
                        $refundData = json_decode($refundResponce, true);

                        if(!empty($refundData)) {
                            $resultData = (object) $refundData;
                            return $resultData;
                        }
                    }
                } else {
                    $requestData = array(
                        'payment_intent' => $paymentIntent
                    );
                    //return Refund Obj
                    $refundResponce = Tools_StripeOAuthTools::createRefundMojo($requestData);

                    if(!empty($refundResponce)) {
                        $refundData = json_decode($refundResponce, true);

                        if(!empty($refundData)) {
                            $resultData = (object) $refundData;
                            return $resultData;
                        }
                    }
                }

                $result['errorMessage'] = $translator->translate('Can not create refund.');
                return $result;
            } catch (Exception $e) {
                $result['errorMessage'] = $e->getMessage();
                return $result;
            }
        } else {
            try {
                Stripe::setApiKey($apiSecretKey);
                return $amount ? Refund::create([
                    'amount' => $amount,
                    'payment_intent' => $paymentIntent,
                ]) : Refund::create([
                    'payment_intent' => $paymentIntent,
                ]);
            } catch (Exception $e) {
                $result['errorMessage'] = $e->getMessage();
                return $result;
            }
        }
    }

    public static function refundCharge($apiSecretKey, $chargeId, $amount = false)
    {
        $result = [];
        $translator = Zend_Registry::get('Zend_Translate');
        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();

        if(!empty($stripepayOAuth)) {
            try {
                if($amount) {
                    $requestData = array(
                        'amount' => $amount,
                        'charge' => $chargeId
                    );
                    //return Refund Obj
                    $refundResponce = Tools_StripeOAuthTools::createRefundMojo($requestData);

                    if(!empty($refundResponce)) {
                        $refundData = json_decode($refundResponce, true);

                        if(!empty($refundData)) {
                            $resultData = (object) $refundData;
                            return $resultData;
                        }
                    }

                } else {
                    $requestData = array(
                        'charge' => $chargeId
                    );
                    //return Refund Obj
                    $refundResponce = Tools_StripeOAuthTools::createRefundMojo($requestData);

                    if(!empty($refundResponce)) {
                        $refundData = json_decode($refundResponce, true);

                        if(!empty($refundData)) {
                            $resultData = (object) $refundData;
                            return $resultData;
                        }
                    }
                }

                $result['errorMessage'] = $translator->translate('Can not create refund.');
                return $result;
            } catch (Exception $e) {
                $result['errorMessage'] = $e->getMessage();
                return $result;
            }
        } else {
            try {
                Stripe::setApiKey($apiSecretKey);
                return $amount ? Refund::create([
                    'amount' => $amount,
                    'charge' => $chargeId,
                ]) : Refund::create([
                    'charge' => $chargeId,
                ]);
            } catch (Exception $e) {
                $result['errorMessage'] = $e->getMessage();
                return $result;
            }
        }
    }
}
