<?php

class Api_Stripepayment_Quotepayment extends Api_Service_Abstract
{


    /**
     * Secure token
     */
    const STRIPEPAY_SECURE_TOKEN = 'StripepayToken';

    /**
     * System response helper
     *
     * @var null
     */
    protected $_responseHelper = null;

    /**
     * Mandatory fields
     *
     * @var array
     */
    protected $_mandatoryParams = array();

    /**
     * User restricted id access
     *
     * @var null
     */
    protected $_restrictedUserId = null;

    /**
     * @var array
     */
    protected $_allowedStatuses = array(
        Models_Model_CartSession::CART_STATUS_NEW,
        Models_Model_CartSession::CART_STATUS_ERROR,
        Models_Model_CartSession::CART_STATUS_PARTIAL
    );

    /**
     * @var array Access Control List
     */
    protected $_accessList = array(
        Tools_Security_Acl::ROLE_SUPERADMIN => array(
            'allow' => array('get', 'post', 'put', 'delete')
        ),
        Tools_Security_Acl::ROLE_ADMIN => array(
            'allow' => array('get', 'post', 'put', 'delete')
        ),
        Shopping::ROLE_SALESPERSON => array(
            'allow' => array('get', 'post', 'put', 'delete')
        ),
        Tools_Security_Acl::ROLE_GUEST => array(
            'allow' => array('get', 'post')
        )
    );

    public function init()
    {
        parent::init();
        $this->_responseHelper = Zend_Controller_Action_HelperBroker::getStaticHelper('response');
        $this->_restrictedUserId = Tools_LeadTools::getUserIdIfActionNotAllowed(Shopping::ROLE_SALESPERSON);
        $this->_loggedUserRoleId = Zend_Controller_Action_HelperBroker::getStaticHelper('session')->getCurrentUser()->getRoleId();

    }


    /**
     *
     * Resource:
     * : /api/stripepayment/quotepayment/
     *
     * HttpMethod:
     * : GET
     *
     * @return JSON
     */
    public function getAction()
    {

        $currency = Zend_Registry::get('Zend_Currency');
        $currencySymbol = preg_replace('~[\w]~', '', $currency->getSymbol());

        $data = array();

        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();
        $privateKey = '';
        $accountId = '';
        if(!empty($stripepayOAuth)) {
            $stripeConfig = Tools_StripeOAuthTools::connectionStatusMojo();
            $stripePublishKey = $stripeConfig['stripe_publishable_key'];
            $accountId = $stripeConfig['stripe_client_id'];
        } else {
            $stripePublishKey = Tools_StripeRegular::getStripeConfig('publishKey');
            $privateKey = Tools_StripeRegular::getStripeConfig('privateKey');
        }

        $stripeSecretKey = Tools_StripeRegular::generatePaymentIntent($privateKey);

        $data['currencyInfo'] = Tools_StripeRegular::getCurrencyInfo();
        $data['additionalInfo'] = array(
            'stripePublishKey' => $stripePublishKey,
            'accountId' => $accountId,
            'currencySymbol' => $currencySymbol
        );
        $data['stripeSecretKey'] = $stripeSecretKey;

        return $data;
    }

    /**
     * Resource:
     * : /api/stripepayment/quotepayment/
     *
     * HttpMethod:
     * : POST
     *
     * @return JSON
     */
    public function postAction()
    {

        $translator = Zend_Registry::get('Zend_Translate');

        $secureToken = $this->getRequest()->getParam(Tools_System_Tools::CSRF_SECURE_TOKEN, false);
        $tokenValid = Tools_System_Tools::validateToken($secureToken, Stripepay::STRIPE_SECURE_TOKEN);
        if (!$tokenValid) {
            $websiteHelper = Zend_Controller_Action_HelperBroker::getStaticHelper('website');
            $websiteUrl = $websiteHelper->getUrl();
            $this->_error($translator->translate('Your session has timed-out. Please Log back in ' . '<a href="' . $websiteUrl . 'go">here</a>'));
        }

        $paymentFirstName = trim(filter_var($this->_request->getParam('paymentFirstName'), FILTER_SANITIZE_STRING, FILTER_FLAG_NO_ENCODE_QUOTES));
        $paymentLastName = trim(filter_var($this->_request->getParam('paymentLastName'), FILTER_SANITIZE_STRING, FILTER_FLAG_NO_ENCODE_QUOTES));
        $stripePaymentMethod = trim(filter_var($this->_request->getParam('stripePaymentMethod'), FILTER_SANITIZE_STRING, FILTER_FLAG_NO_ENCODE_QUOTES));
        $quotePageId = trim(filter_var($this->_request->getParam('quotePageId'), FILTER_SANITIZE_NUMBER_INT));

        $privateKey = Tools_StripeRegular::getStripeConfig('privateKey');

        if (empty($stripePaymentMethod)) {
            return array(
                'error' => '1',
                'message' => $translator->translate(Stripepay::TRANSLATION_PREFIX . 'payment_system_error')
            );
        }

        $quoteMapper = Quote_Models_Mapper_QuoteMapper::getInstance();
        $cartStorage = Tools_ShoppingCart::getInstance();
        $cartId = $cartStorage->getCartId();
        $cartSessionMapper = Models_Mapper_CartSessionMapper::getInstance();
        $cart = $cartSessionMapper->find($cartId);
        if ($cart instanceof Models_Model_CartSession) {
            $cartStatus = $cart->getStatus();
            $gateway = $cart->getGateway();
            if (!in_array($cartStatus, $this->_allowedStatuses) && $gateway == 'Stripepay') {
                return array(
                    'error' => '1',
                    'message' => $translator->translate('We can\'t confirm your order at the moment due to a technical difficulty. If you do not receive an email in the coming hours confirming your purchase, please contact us')
                );
            }
        }

        $quoteModel = $quoteMapper->findByCartId($cart->getId());
        if (!$quoteModel instanceof Quote_Models_Model_Quote) {
            return array(
                'error' => 1,
                'generalError' => $translator->translate('You can pay only on quote!')
            );
        }

        $currentQuotePageId = $quoteModel->getId();
        $quotePageUrl = $currentQuotePageId . '.html';
        $quotePageModel = Application_Model_Mappers_PageMapper::getInstance()->findByUrl($quotePageUrl);
        if ((int)$quotePageModel->getId() !== (int) $quotePageId) {
            return array(
                'error' => '1',
                'message' => $translator->translate('Wrong quote')
            );
        }

        $signatureRequired = $quoteModel->getIsSignatureRequired();
        if (!empty($signatureRequired) && empty($quoteModel->getIsQuoteSigned())) {
            return array(
                'error' => '1',
                'message' => $translator->translate('Please sign the quote!')
            );
        }


        $quoteModel = $quoteMapper->findByCartId($cart->getId());
        if (!$quoteModel instanceof Quote_Models_Model_Quote) {
            $this->_responseHelper->fail(array(
                'error' => 1,
                'generalError' => $translator->translate('You can pay only on quote!')
            ));
        }

        $websiteHelper = Zend_Controller_Action_HelperBroker::getStaticHelper('website');
        $websiteUrl = $websiteHelper->getUrl();
        $redirectUrl = $websiteUrl . $quoteModel->getId() . '.html';

        $isPartial = false;
        if (!empty($cart->getIsPartial())) {
            $isPartial = true;
        }

        $partialPaymentType = $cart->getPartialType();

        $updatePaymentStatus = Models_Model_CartSession::CART_STATUS_COMPLETED;
        if ($isPartial === true) {
            $partialPaidAmount = $cart->getPartialPaidAmount();
            $cart->setIsPartial('1');
            if (!empty($partialPaidAmount) && $partialPaidAmount !== '0.00' && !empty((int)$partialPaidAmount)) {
                $cart->setPartialPaidAmount($cart->getTotal());
                if ($partialPaymentType === Models_Model_CartSession::CART_PARTIAL_PAYMENT_TYPE_AMOUNT) {
                    $amountToPayPartial = round($cart->getTotal() - $cart->getPartialPercentage(), 2);
                } else {
                    $amountToPayPartial = round($cart->getTotal() - round(($cart->getTotal() * $cart->getPartialPercentage()) / 100,
                            2), 2);
                }

                $updatePaymentStatus = Models_Model_CartSession::CART_STATUS_COMPLETED;
            } else {
                if ($partialPaymentType === Models_Model_CartSession::CART_PARTIAL_PAYMENT_TYPE_AMOUNT) {
                    $amountToPayPartial = round($cart->getPartialPercentage(), 2);
                } else {
                    $amountToPayPartial = round(($cart->getTotal() * $cart->getPartialPercentage()) / 100, 2);
                }

                $cart->setPartialPaidAmount($amountToPayPartial);
                $updatePaymentStatus = Models_Model_CartSession::CART_STATUS_PARTIAL;
            }
        }

        $cartSummary = array();
        $cartSummary['totalTax'] = round($cart->getTotalTax(), 2);
        $cartSummary['shipping'] = round($cart->getShippingPrice(), 2);
        $cartSummary['total'] = round($cart->getTotal(), 2);

        $amountToPay = $cartSummary['total'];
        if ($isPartial === true && !empty($amountToPayPartial)) {
            $amountToPay = $amountToPayPartial;
        }

        $userId = $quoteModel->getUserId();
        $userModel = Application_Model_Mappers_UserMapper::getInstance()->find($userId);
        if (!$userModel instanceof Application_Model_Models_User) {
            return array(
                'error' => 1,
                'generalError' => 'User not found'
            );
        }

        $customerId = $userModel->getId();

        $customerShipping = '';
        $shippingAddressKey = Tools_ShoppingCart::getInstance()->getShippingAddressKey();
        if ($shippingAddressKey != null) {
            $customerShipping = Tools_ShoppingCart::getAddressById($shippingAddressKey);
        }

        $billingAddressArray = array();
        $billingAddressArray['firstname'] = $paymentFirstName;
        $billingAddressArray['lastname'] = $paymentLastName;
        $billingAddressArray['company'] = '';
        $billingAddressArray['email'] = $userModel->getEmail();
        $billingAddressArray['address1'] = '';
        $billingAddressArray['address2'] = '';
        $billingAddressArray['city'] = '';
        $billingAddressArray['zip'] = '';
        $billingAddressArray['phone'] = '';
        $billingAddressArray['phone_country_code_value'] = '';
        $billingAddressArray['phonecountrycode'] = '';

        $metaData = array();
        $metaData['cartId'] = $cartId;
        $metaData['Customer name'] = $paymentFirstName . ' ' . $paymentLastName;
        $metaData['userEmail'] = $userModel->getEmail();
        $metaData['amount'] = $amountToPay;
        $metaData['userFirstName'] = $paymentFirstName;
        $metaData['userLastName'] = $paymentLastName;
        $metaData['totalTax'] = $cartSummary['totalTax'];
        $metaData['customerShipping'] = $customerShipping ? json_encode($customerShipping) : '';
        $metaData['billingAddress'] = json_encode($billingAddressArray);

        $stripePay = new Stripepay(array(), array());
        $stripePay->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_PROCESSING, true);

        $gw = new Tools_StripeRecurrent($privateKey, $stripePaymentMethod, $metaData);
        $payResult = $gw->processPaymentIntent(Stripepay::PAYMENT_TYPE_CART_QUOTE);

        if (!empty($payResult['error'])) {
            $stripePay->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_ERROR, true, $payResult['message']);
            return array(
                'error' => '1',
                'message' => $payResult['message']
            );
        }

        $cartSessionMapper->save($cart);
        $stripePay->updateCartStatus($cartId, $updatePaymentStatus);

        $cartSession = $cartSessionMapper->find($cartId);
        $sessionHelper = Zend_Controller_Action_HelperBroker::getStaticHelper('session');
        $sessionHelper->storeCartSessionKey = $cartId;
        $sessionHelper->storeCartSessionConversionKey = $cartId;
        if ($cartSession->getStatus() !== Models_Model_CartSession::CART_STATUS_PARTIAL) {
            if ($isPartial === true) {
                $cartSession->registerObserver(new Tools_Mail_Watchdog(array(
                    'trigger' => Tools_StoreMailWatchdog::TRIGGER_STORE_PARTIALPAYMENT_SECOND
                )));
            } else {
                $cartSession->registerObserver(new Tools_Mail_Watchdog(array(
                    'trigger' => Tools_StoreMailWatchdog::TRIGGER_NEW_ORDER
                )));
            }
        }

        $shoppingConfig = Models_Mapper_ShoppingConfig::getInstance()->getConfigParams();

        if (!empty($shoppingConfig[Shopping::SHIPPING_IS_GIFT])) {
            if (!empty($cartSession->getIsGift())) {
                $cartSession->registerObserver(new Tools_Mail_Watchdog(array(
                    'trigger' => Tools_StoreMailWatchdog::TRIGGER_STORE_GIFT_ORDER
                )));
            }
        }

        if ($cartSession->getStatus() === Models_Model_CartSession::CART_STATUS_PARTIAL) {
            $cartSession->registerObserver(new Tools_Mail_Watchdog(array(
                'trigger' => Tools_StoreMailWatchdog::TRIGGER_STORE_PARTIALPAYMENT
            )));
        }

        if ($cartSession->getStatus() !== Models_Model_CartSession::CART_STATUS_PARTIAL) {
            if (class_exists('Tools_AppsServiceWatchdog')) {
                $cartSession->registerObserver(new Tools_AppsServiceWatchdog());
            }
        }

        $cartSession->notifyObservers();

        $currency = Zend_Registry::get('Zend_Currency');
        if (empty($amountToPayPartial)) {
            $message = $currency->toCurrency($amountToPay);
        } else {
            $message = $currency->toCurrency($amountToPayPartial);
        }

        return array(
            'error' => '0',
            'message' => $translator->translate('Thank you for your payment of ') . $message,
            'redirectUrl' => $redirectUrl
        );

    }

    /**
     *
     * Resource:
     *
     * HttpMethod:
     * : PUT
     *
     * ## Parameters:
     * id (source integer)
     *
     * @return JSON
     */
    public function putAction()
    {

    }

    /**
     *
     * Resource:
     *
     * HttpMethod:
     * : DELETE
     *
     * ## Parameters:
     * id (source integer)
     *
     * @return JSON
     */
    public function deleteAction()
    {


    }

}
