<?php

require __DIR__ . '/vendor/autoload.php';

use Stripe\Stripe;
use Stripe\Checkout\Session;
use Stripe\Webhook;
use Stripe\Exception;
use Stripe\SetupIntent;
use Stripe\Refund;
use Stripe\PaymentIntent;
use Stripe\ApplePayDomain;

class Stripepay extends Tools_PaymentGateway
{

    const TRANSLATION_PREFIX = 'stripepay_';

    protected $_layout = null;

    protected $_translator = null;

    private $_websiteConfig;

    protected $_allowedStatuses = array(Models_Model_CartSession::CART_STATUS_NEW, Models_Model_CartSession::CART_STATUS_ERROR);

    protected $_allowedStatusesCheckout = array(
        Models_Model_CartSession::CART_STATUS_NEW,
        Models_Model_CartSession::CART_STATUS_ERROR,
        Models_Model_CartSession::CART_STATUS_PROCESSING
    );

    protected $_pdfPath;

    protected $_allowedStatusesForNewOrder = array(
        Models_Model_CartSession::CART_STATUS_NEW,
        Models_Model_CartSession::CART_STATUS_ERROR,
        Models_Model_CartSession::CART_STATUS_PARTIAL
    );

    protected $_googleApplePayAllowedCountires = array(
        'AE',
        'AT',
        'AU',
        'BE',
        'BG',
        'BR',
        'CA',
        'CH',
        'CI',
        'CR',
        'CY',
        'CZ',
        'DE',
        'DK',
        'DO',
        'EE',
        'ES',
        'FI',
        'FR',
        'GB',
        'GI',
        'GR',
        'GT',
        'HK',
        'HR',
        'HU',
        'ID',
        'IE',
        'IN',
        'IT',
        'JP',
        'LI',
        'LT',
        'LU',
        'LV',
        'MT',
        'MX',
        'MY',
        'NL',
        'NO',
        'NZ',
        'PE',
        'PH',
        'PL',
        'PT',
        'RO',
        'SE',
        'SG',
        'SI',
        'SK',
        'SN',
        'TH',
        'TT',
        'US',
        'UY'
    );

    /**
     * Secure token
     */
    const STRIPE_SECURE_TOKEN = 'StripepayToken';
    const ACTIVE_STATUS = 'active';
    const TRANSACTION_TYPE_SESSION_CHECKOUT = 'session_checkout';
    const TRANSACTION_STATUS_PROCESSING = 'processing';
    const TRANSACTION_STATUS_SUCCESS = 'succeeded';

    const PAYMENT_TYPE_BUTTON = 'button';
    const PAYMENT_TYPE_CART = 'cart';
    const PAYMENT_TYPE_RECURRENT = 'recurrent';
    const PAYMENT_TYPE_SINGLE_PRODUCT = 'single_product';
    const PAYMENT_TYPE_CART_QUOTE = 'cart_quote';
    const ERROR_PRODUCT_NOT_FOUND = 'Product not found';
    const ERROR_CUSTOMER_NOT_FOUND = 'Customer not found';
    const ERROR_PAYMENT_METHOD_NOT_FOUND = 'Payment method not found.';
    const ERROR_PAYMENT_CONNECTION_ERROR = 'Gateway connection error';
    const ERROR_PAYMENT_COMMON = 'Something went wrong.';

    const SINGLE_PRODUCT_RECURRING_PERIODS = [
        '1MONTH' => 'MONTH',
        '2MONTH' => 'TWO MONTHS',
        '3MONTH' => 'THREE MONTHS',
        '6MONTH' => 'SIX MONTHS',
        '1YEAR' => 'YEAR',
    ];
    const MONTH_PERIOD = 'month';
    const YEAR_PERIOD = 'year';

    const APPLE_PAY_DIR = '.well-known';
    const DOMAIN_ASSOCIATION_FILE_URL = 'https://stripe.com/files/apple-pay/apple-developer-merchantid-domain-association';
    const DOMAIN_ASSOCIATION_FILE_NAME = 'apple-developer-merchantid-domain-association';

    public static function pluginApiActions()
    {
        return array(
            'mojoSessionEventSuccessPurchase',
        );
    }

    public function __construct($options, $seotoasterData)
    {
        parent::__construct($options, $seotoasterData);

        $this->_layout = new Zend_Layout();
        $this->_layout->setLayoutPath(Zend_Layout::getMvcInstance()->getLayoutPath());

        if ($viewScriptPath = Zend_Layout::getMvcInstance()->getView()->getScriptPaths()) {
            $this->_view->setScriptPath($viewScriptPath);
        }
        $this->_view->addScriptPath(__DIR__ . '/system/views/');

        $this->_translator = Zend_Registry::get('Zend_Translate');
        $this->_jsonHelper = Zend_Controller_Action_HelperBroker::getStaticHelper('json');
        $this->_configHelper = Zend_Controller_Action_HelperBroker::getStaticHelper('config');
        $this->_websiteConfig = Zend_Registry::get('website');
        $this->_pdfPath = $this->_websiteConfig['path'] . 'plugins' . DIRECTORY_SEPARATOR . 'invoicetopdf' . DIRECTORY_SEPARATOR . 'invoices' . DIRECTORY_SEPARATOR;


    }

    public function beforeController()
    {
        //self::mojoSessionEventSuccessPurchase(array());
    }

    public function run($requestedParams = array()) {
        $dispatchersResult = parent::run($requestedParams);
        if($dispatchersResult) {
            return $dispatchersResult;
        }

    }

    /**
     * Returns the array of Stripe config options
     * @return array
     **/
    protected function _getStripeConfig()
    {
        $stripeConfigMapper = Stripepay_Models_Mappers_StripepayConfigMapper::getInstance();
        $stripeConfig = $stripeConfigMapper->getConfigParams();
        return [
            'privateKey' => $stripeConfig['useSandBox'] == "0" ? $stripeConfig['privateKey'] : $stripeConfig['privateKeyTest'],
            'publishKey' => $stripeConfig['useSandBox'] == "0" ? $stripeConfig['publishableKey'] : $stripeConfig['publishableKeyTest'],
            'useSandBox' => isset($stripeConfig['useSandBox']) ? $stripeConfig['useSandBox'] == "1" : true,
            'disableLink' => isset($stripeConfig['disableLink']) ? $stripeConfig['disableLink'] == "1" : true,
            'endpointSecret' => isset($stripeConfig['endpointSecret']) ? $stripeConfig['endpointSecret'] : ''
        ];
    }

    protected function configAction()
    {
        if (Tools_Security_Acl::isAllowed(Tools_Security_Acl::RESOURCE_PLUGINS)) {
            $stripeConfigMapper = Stripepay_Models_Mappers_StripepayConfigMapper::getInstance();
            $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();

            if ($this->_request->isPost()) {
                $configParams = filter_var_array($this->_request->getParams(), FILTER_SANITIZE_STRING);

                $publishableKeyTest = $configParams['publishableKeyTest'];
                $privateKeyTest = $configParams['privateKeyTest'];
                $publishableKey = $configParams['publishableKey'];
                $privateKey = $configParams['privateKey'];
                $endpointSecret = $configParams['endpointSecret'];

                $secureToken = $this->_request->getParam(Tools_System_Tools::CSRF_SECURE_TOKEN, false);

                $tokenValid = Tools_System_Tools::validateToken($secureToken, self::STRIPE_SECURE_TOKEN);
                if (!$tokenValid) {
                    $this->_responseHelper->fail('');
                }

                if(empty($stripepayOAuth)) {
                    $stripeConfig = $stripeConfigMapper->getConfigParams();

                    if (empty($publishableKeyTest)) {
                        $configParams['publishableKeyTest'] = '';
                        if (isset($stripeConfig['publishableKeyTest'])) {
                            $configParams['publishableKeyTest'] = $stripeConfig['publishableKeyTest'];
                        }
                    }

                    if (empty($privateKeyTest)) {
                        $configParams['privateKeyTest'] = '';
                        if (isset($stripeConfig['privateKeyTest'])) {
                            $configParams['privateKeyTest'] = $stripeConfig['privateKeyTest'];
                        }
                    }

                    if (empty($publishableKey)) {
                        $configParams['publishableKey'] = '';
                        if (isset($stripeConfig['publishableKey'])) {
                            $configParams['publishableKey'] = $stripeConfig['publishableKey'];
                        }
                    }

                    if (empty($privateKey)) {
                        $configParams['privateKey'] = '';
                        if (isset($stripeConfig['privateKey'])) {
                            $configParams['privateKey'] = $stripeConfig['privateKey'];
                        }
                    }

                    if (empty($endpointSecret)) {
                        $configParams['endpointSecret'] = '';
                        if (isset($stripeConfig['endpointSecret'])) {
                            $configParams['endpointSecret'] = $stripeConfig['endpointSecret'];
                        }
                    }

                    if (!empty($configParams['useSandBox'])) {
                        if (empty($publishableKeyTest) || empty($configParams['privateKeyTest'])) {
                            $this->_responseHelper->fail($this->_translator->translate('Stripe not configured!'));
                        }
                    } else {
                        if (empty($configParams['privateKey']) || empty($publishableKey)) {
                            $this->_responseHelper->fail($this->_translator->translate('Stripe not configured!'));
                        }
                    }
                }

                $stripeConfigMapper->save($configParams);

                $this->_responseHelper->success('');
            } else {
                if(!empty($stripepayOAuth)) {
                    $this->_view->stripepayOAuth = $stripepayOAuth;

                    //return array
                    $stripepayConnectionStatus = Tools_StripeOAuthTools::connectionStatusMojo();
                    $failResponce = false;
                    if(empty($stripepayConnectionStatus['done'])) {
                        $failResponce = true;
                    }

                    $this->_view->failResponce = $failResponce;

                    $connectStatus = $this->_translator->translate('Disconnected');
                    $isConnected = 0;
                    if(!empty($stripepayConnectionStatus['status'])) {
                        if($stripepayConnectionStatus['status'] == 'connected') {
                            $isConnected = 1;
                            $connectStatus = $this->_translator->translate('Connected');
                        }
                    }
                    $mode = 'production';
                    if(!empty($stripepayConnectionStatus['mode'])) {
                        $mode = $stripepayConnectionStatus['mode'];
                    }

                    $this->_view->mode = $mode;
                    $this->_view->isConnected = $isConnected;
                    $this->_view->connectStatus = $connectStatus;
                    $this->_view->authorizationCodeURL = $stripepayConnectionStatus['oauthLink'];
                }

                $stripeConfig = $stripeConfigMapper->getConfigParams();
                $this->_view->settings = $stripeConfig;

                $this->_view->translator = $this->_translator;
                $this->_layout->content = $this->_view->render('stripeConfig.phtml');
                echo $this->_layout->render();
            }
        }
    }

    public function getOAuthConfigDetailsAction()
    {
        $tokenToValidate = $this->_request->getParam(Tools_System_Tools::CSRF_SECURE_TOKEN, false);
        $valid = Tools_System_Tools::validateToken($tokenToValidate, self::STRIPE_SECURE_TOKEN);
        if (!$valid) {
            $this->_responseHelper->fail('');
        }

        $stripepayConnectionStatus = Tools_StripeOAuthTools::connectionStatusMojo();

        if(empty($stripepayConnectionStatus['done'])) {
            $this->_responseHelper->fail('');
        }

        if($stripepayConnectionStatus['status'] == 'connected') {
            $this->_responseHelper->success(array('status' => 'connected', 'mode' => $stripepayConnectionStatus['mode']));
        } elseif ($stripepayConnectionStatus['status'] == 'disconnected') {
            $this->_responseHelper->success(array('status' => 'disconnected', 'mode' => $stripepayConnectionStatus['mode']));
        }

        $this->_responseHelper->fail('');
    }

    public function downloadDomainAssociationFileAction()
    {
        if ($this->_request->isPost()) {
            $tokenToValidate = $this->_request->getParam(Tools_System_Tools::CSRF_SECURE_TOKEN, false);
            $valid = Tools_System_Tools::validateToken($tokenToValidate, self::STRIPE_SECURE_TOKEN);
            if (!$valid) {
                $this->_responseHelper->fail('');
            }

            $websitePath = $this->_websiteHelper->getPath();
            $applePayDir = $websitePath . self::APPLE_PAY_DIR;

            if (!is_dir($applePayDir)) {
                try {
                    Tools_Filesystem_Tools::mkDir($applePayDir);
                } catch (Exceptions_SeotoasterException $e) {
                    $this->_responseHelper->fail($this->_translator->translate('Can\'t create dir '.self::APPLE_PAY_DIR));
                }
            }

            if(is_dir($applePayDir)) {
                $url  = self::DOMAIN_ASSOCIATION_FILE_URL;
                $fp = fopen ($websitePath . self::APPLE_PAY_DIR . DIRECTORY_SEPARATOR . self::DOMAIN_ASSOCIATION_FILE_NAME, 'w+');
                $ch = curl_init(str_replace(" ","%20",$url));
                curl_setopt($ch, CURLOPT_TIMEOUT, 600);
                curl_setopt($ch, CURLOPT_FILE, $fp);
                curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
                $response = curl_exec($ch);
                curl_close($ch);
                fclose($fp);

                if($response) {
                    $this->_responseHelper->success($this->_translator->translate(self::DOMAIN_ASSOCIATION_FILE_NAME . ' successfully downloaded.'));
                } else {
                    $this->_responseHelper->fail($this->_translator->translate('Can\'t download '. self::DOMAIN_ASSOCIATION_FILE_NAME));
                }
            }
        }
    }

    public function setModeAction()
    {
        $tokenToValidate = $this->_request->getParam(Tools_System_Tools::CSRF_SECURE_TOKEN, false);
        $valid = Tools_System_Tools::validateToken($tokenToValidate, self::STRIPE_SECURE_TOKEN);
        if (!$valid) {
            $this->_responseHelper->fail('');
        }

        $mode = filter_var($this->_request->getParam('mode'), FILTER_SANITIZE_STRING);

        if(!empty($mode)) {
            $modeToRequest = 0;
            if($mode == 'test') {
                $modeToRequest = 1;
            }

            $stripeSetTestMode = Tools_StripeOAuthTools::setTestModeMojo(array('testMode' => $modeToRequest));
            //$stripeSetTestMode['done'] = true;

            if(!empty($stripeSetTestMode['done'])) {
                $stripepayConnectionStatus = Tools_StripeOAuthTools::connectionStatusMojo();

//                $stripepayConnectionStatus = array (
//                        'done' => true,//false
//                        'status' => 'disconnected',
//                        'oauthLink' => 'https://aleksfromiphone-seven.diy.seosamba.com/',
//                        'stripe_publishable_key' => 'pk_live_***',
//                        'stripe_client_id' => 'acct_***',
//                        'mode' => 'production'//production
//                );

                if(empty($stripepayConnectionStatus['done'])) {
                    $this->_responseHelper->fail('');
                }

                $connectStatus = $this->_translator->translate('Disconnected');
                $isConnected = 0;
                if(!empty($stripepayConnectionStatus['status'])) {
                    if($stripepayConnectionStatus['status'] == 'connected') {
                        $isConnected = 1;
                        $connectStatus = $this->_translator->translate('Connected');
                    }
                }
                $mode = 'production';
                if(!empty($stripepayConnectionStatus['mode'])) {
                    $mode = $stripepayConnectionStatus['mode'];
                }

                $this->_responseHelper->success(array(
                    'mode' => $mode,
                    'isConnected' => $isConnected,
                    'connectStatus' => $connectStatus,
                    'authorizationCodeURL' => $stripepayConnectionStatus['oauthLink'],
                ));

            }
        }

        $this->_responseHelper->fail('');
    }

    public function _makeOptionCreditcard()
    {
        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();
        if(!empty($stripepayOAuth)) {
            $stripeConfig = Tools_StripeOAuthTools::connectionStatusMojo();
            $publishKey = $stripeConfig['stripe_publishable_key'];
            $accountId = $stripeConfig['stripe_client_id'];
        } else {
            $stripeConfig = $this->_getStripeConfig();
            $publishKey = $stripeConfig['publishKey'];
            $privateKey = $stripeConfig['privateKey'];
        }

        if ((empty($privateKey) || empty($publishKey)) && empty($stripepayOAuth)) {
            return $this->_translator->translate('Stripe not configured!');
        }
        $cartContent = Tools_ShoppingCart::getInstance();
        $productList = array_values($cartContent->getContent());
        $this->_view->cartTotal = $cartContent->getTotal();
        $this->_view->translator = $this->_translator;
        $shoppingConfigMapper = Models_Mapper_ShoppingConfig::getInstance();

        if (!empty($productList)) {
            $this->_view->productList = $productList;
            $cartParams = $cartContent->calculate();
            $this->_view->totalTax = round($cartParams['totalTax'], 2);
            $this->_view->shipping = round($cartParams['shipping'], 2);
            $this->_view->totalAmount = round($cartParams['total'], 2);
            $this->_view->currency = $shoppingConfigMapper->getConfigParam('currency');
            $shippingAddressKey = Tools_ShoppingCart::getInstance()->getShippingAddressKey();
            if ($shippingAddressKey != null) {
                $this->_view->allowShipping = true;
            }
            $this->_view->countryList = $this->_prepareCountryList();
            $country = $shoppingConfigMapper->getConfigParam('country');
            if ($country != '') {
                $predefinedCountry = $country;
            } else {
                $predefinedCountry = 'US';
            }
            $phonecountrycodes = Tools_System_Tools::getFullCountryPhoneCodesList(true, array(), true);
            $phonecountrycode = $predefinedCountry;
            $listMasksMapper = Application_Model_Mappers_MasksListMapper::getInstance();
            $this->_view->phonecountrycodes = $phonecountrycodes;
            $this->_view->desktopMasks = $listMasksMapper->getListOfMasksByType(Application_Model_Models_MaskList::MASK_TYPE_DESKTOP);
            $billingAddressKey = Tools_ShoppingCart::getInstance()->getBillingAddressKey();
            if ($billingAddressKey != null) {
                $paymentParams = Tools_ShoppingCart::getAddressById($billingAddressKey);
                if (isset($paymentParams['state'])) {
                    $state = Tools_Geo::getStateById($paymentParams['state']);
                    $paymentParams['state'] = $state['id'];
                    $this->_view->state = $paymentParams['state'];
                }
                $phonecountrycode = $paymentParams['phonecountrycode'];
                $this->_view->firstName = $paymentParams['firstname'];
                $this->_view->lastName = $paymentParams['lastname'];
                $this->_view->address1 = $paymentParams['address1'];
                $this->_view->address2 = $paymentParams['address2'];
                $this->_view->billingEmail = $paymentParams['email'];
                $this->_view->city = $paymentParams['city'];
                $this->_view->zip = $paymentParams['zip'];
                $this->_view->phone = $paymentParams['phone'];
                $predefinedCountry = $paymentParams['country'];
            }
            $this->_view->phonecountrycode = $phonecountrycode;
            $listOfStates = $this->prepareStateList($predefinedCountry);
            if ($listOfStates['error'] == '0') {
                $this->_view->listOfState = $listOfStates[0];
            }
            $cartId = $cartContent->getCartId();
            $cart = Models_Mapper_CartSessionMapper::getInstance()->find($cartId);
            if ($cart instanceof Models_Model_CartSession) {
                $gateway = $cart->getGateway();
                $cartStatus = $cart->getStatus();
                if (!in_array($cartStatus, $this->_allowedStatuses) && $gateway == 'Stripepay') {
                    $this->_view->cartStatusError = true;
                    unset($this->_sessionHelper->storeCartSessionKey);
                    unset($this->_sessionHelper->storeCartSessionConversionKey);
                    $cartContent->clean();
                }
            }
            $this->_view->predefinedCountry = $predefinedCountry;
            $this->_view->publishKey = $publishKey;
            $this->_view->translator = $this->_translator;

            $recurrent = $cartContent->getRecurringPaymentType();

            $this->_view->accountId = '';
            if (!empty($recurrent)) {
                $intent = Tools_RecurringPaymentStripepay::createSetupIntent($privateKey);

                if(!empty($intent['intent'])) {
                    $intentData = json_decode($intent['intent'], true);

                    if(!empty($intentData)) {
                        $intent = (object) $intentData;
                        $this->_view->accountId = $accountId;
                    }
                }

                if (!is_object($intent)) {
                    return $this->_translator->translate('Stripe intent cannot be created!');
                }
            } else {
                $dataRequest = array(
                    'amount' => $this->_view->totalAmount * 100,
                    'currency' => strtolower($this->_view->currency),//'usd',
                    'payment_method_types' => ['card'],
                    'metadata' => ['cartId' => $cartId]
                );

                if(!empty($stripepayOAuth)) {
                    $configData = Application_Model_Mappers_ConfigMapper::getInstance()->getConfig();
                    $websiteId = $configData['websiteId'];
                    $dataRequest['metadata']['websiteId'] = $websiteId;

                    $responcePaymentIntent = Tools_StripeOAuthTools::createPaymentIntentMojo($dataRequest);

                    if(!empty($responcePaymentIntent)) {
                        $responcePaymentIntentData = json_decode($responcePaymentIntent, true);

                        if(!empty($responcePaymentIntentData)) {
                            $intent = (object) $responcePaymentIntentData;
                            $this->_view->accountId = $accountId;
                        }
                    }
                } else {
                    Stripe::setApiKey($privateKey);
                    $intent = PaymentIntent::create($dataRequest);
                }

                if (!is_object($intent)) {
                    return $this->_translator->translate('Stripe intent cannot be created!');
                }
            }

            $this->_view->intent = $intent;

            return $this->_view->render('stripeForm.phtml');
        }
    }

    protected function _makeOptionCreditcarrdquote()
    {
        if (Tools_Security_Acl::isAllowed(Shopping::RESOURCE_STORE_MANAGEMENT)) {
            return '';
        }

        $privateKeyTest = '';
        $publishableKeyTest = '';
        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();
        if(empty($stripepayOAuth)) {
            $privateKeyTest = Tools_StripeRegular::getStripeConfig('privateKey');
            $publishableKeyTest = Tools_StripeRegular::getStripeConfig('publishKey');
        }

        if ((!empty($privateKeyTest) && !empty($publishableKeyTest)) || !empty($stripepayOAuth)) {

            $pageHelper = Zend_Controller_Action_HelperBroker::getStaticHelper('page');
            $front = Zend_Controller_Front::getInstance();
            $productMapper = Models_Mapper_ProductMapper::getInstance();
            $pageMapper = Application_Model_Mappers_PageMapper::getInstance();
            $quote = Quote_Models_Mapper_QuoteMapper::getInstance()->find(
                $pageHelper->clean($front->getRequest()->getParams('page'))
            );
            if (!$quote instanceof Quote_Models_Model_Quote) {
                return '';
            }
            $cart = Models_Mapper_CartSessionMapper::getInstance()->find($quote->getCartId());
            if (!$cart instanceof Models_Model_CartSession) {
                return '';
            }
            if ($quote->getStatus() === Quote_Models_Model_Quote::STATUS_LOST) {
                return '';
            }

            $pageModel = $pageMapper->findByUrl($quote->getId().'.html');
            if (!$pageModel instanceof Application_Model_Models_Page) {
                return null;
            }

            $this->_view->pageId = $pageModel->getId();

            $recalculationFlags = $this->_setRecalculationFlags();
            if (!empty($quote->getCreatorId())) {
                $recalculationFlags = $this->_setRecalculationFlags($quote->getCreatorId());
            }
            if (!empty($quote->getEditorId())) {
                $recalculationFlags = $this->_setRecalculationFlags($quote->getEditorId());
            }
            $cartContent = $cart->getCartContent();
            $discount = $cart->getDiscount();
            $discountRate = $cart->getDiscountTaxRate();
            if (!$cartContent) {
                return null;
            }

            $status = $cart->getStatus();

            if ($status != Models_Model_CartSession::CART_STATUS_COMPLETED && $status != Models_Model_CartSession::CART_STATUS_DELIVERED && $status != Models_Model_CartSession::CART_STATUS_SHIPPED) {
                if ($status === Models_Model_CartSession::CART_STATUS_PENDING && $cart->getGateway() !== 'Quote') {
                    return '';
                }

                $cartSession = Tools_ShoppingCart::getInstance();
                $cartSession->setContent(array());
                $cartSession->save();
                $cartSession->setCartId($quote->getCartId());
                $cartSession->setShippingAddressKey($cart->getShippingAddressId());
                $cartSession->setBillingAddressKey($cart->getBillingAddressId());
                $notFreebiesInCart = array();
                $freebiesInCart = array();
                $productsFreebiesRelation = array();
                foreach ($cartContent as $key => $product) {
                    if ($product['freebies'] === '1') {
                        $freebiesInCart[$product['product_id']] = $product['product_id'];
                    } else {
                        $notFreebiesInCart[$product['product_id']] = $product['product_id'];
                    }
                }
                if (!empty($freebiesInCart)) {
                    $where = $productMapper->getDbTable()->getAdapter()->quoteInto(
                        'sphp.freebies_id IN (?)',
                        $freebiesInCart
                    );
                    $where .= ' AND ' . $productMapper->getDbTable()->getAdapter()->quoteInto(
                            'sphp.product_id IN (?)',
                            $notFreebiesInCart
                        );
                    $select = $productMapper->getDbTable()->getAdapter()->select()
                        ->from(
                            array('spfs' => 'shopping_product_freebies_settings'),
                            array(
                                'freebiesGroupKey' => new Zend_Db_Expr("CONCAT(sphp.freebies_id, '_', sphp.product_id)"),
                                'price_value'
                            )
                        )
                        ->joinleft(
                            array('sphp' => 'shopping_product_has_freebies'),
                            'spfs.prod_id = sphp.product_id'
                        )
                        ->where($where);
                    $productFreebiesSettings = $productMapper->getDbTable()->getAdapter()->fetchAssoc($select);
                }

                if (!empty($productFreebiesSettings)) {
                    foreach ($productFreebiesSettings as $prodInfo) {
                        if (array_key_exists($prodInfo['freebies_id'], $freebiesInCart)) {
                            if (isset($productsFreebiesRelation[$prodInfo['freebies_id']])) {
                                $productsFreebiesRelation[$prodInfo['freebies_id']][$prodInfo['product_id']] = $prodInfo['product_id'];
                            } else {
                                $productsFreebiesRelation[$prodInfo['freebies_id']] = array($prodInfo['product_id'] => $prodInfo['product_id']);
                            }
                        }
                    }
                }

                foreach ($cartContent as $key => $product) {
                    $productObject = $productMapper->find($product['product_id']);
                    if ($productObject instanceof Models_Model_Product) {
                        if ($product['freebies'] === '1' && !empty($productsFreebiesRelation)) {
                            foreach ($productsFreebiesRelation[$product['product_id']] as $realProductId) {
                                $itemKey = $this->_generateStorageKey(
                                    $productObject,
                                    array(0 => 'freebies_' . $realProductId)
                                );
                                if (!$cartSession->findBySid($itemKey)) {
                                    $productObject->setFreebies(1);
                                    $cartSession->add(
                                        $productObject,
                                        array(0 => 'freebies_' . $realProductId),
                                        $product['qty'], true, $recalculationFlags['skipOptionRecalculation'],
                                        $recalculationFlags['skipGroupPriceRecalculation']
                                    );
                                }
                            }
                        } else {
                            $options = array();
                            if (is_array($product['options'])) {
                                $options = $this->_parseProductOptions($product['options']);
                            }
                            $productObject->setPrice($product['price']);
                            $productObject->setOriginalPrice($product['original_price']);
                            $productObject->setCurrentPrice(floatval($productObject->getPrice()));
                            $cartSession->add($productObject, $options, $product['qty'], true,
                                $recalculationFlags['skipOptionRecalculation'],
                                $recalculationFlags['skipGroupPriceRecalculation']);
                        }
                    }
                }
                $cartSession->setDiscount($discount);
                $cartSession->setShippingData(array('price' => $cart->getShippingPrice()));
                $cartSession->setDiscountTaxRate($discountRate);
                $cartSession->calculate(true, $recalculationFlags['skipGroupPriceRecalculation']);
                $cartSession->save();

                $this->_view->partialPayment = false;
                $paymentType = $quote->getPaymentType();
                if ($paymentType === Quote_Models_Model_Quote::PAYMENT_TYPE_PARTIAL_PAYMENT) {
                    $cartStatus = $cart->getStatus();
                    $partialPercent = $cart->getPartialPercentage();
                    $partialPaymentType = $cart->getPartialType();
                    if (($cartStatus === Models_Model_CartSession::CART_STATUS_PARTIAL || $cartStatus === Models_Model_CartSession::CART_STATUS_PROCESSING) && !empty((int)$cart->getPartialPaidAmount())) {
                        if ($partialPaymentType === Models_Model_CartSession::CART_PARTIAL_PAYMENT_TYPE_AMOUNT) {
                            $partialPercent = round($partialPercent, 6);
                        } else {
                            $partialPercent = round(100 - $partialPercent, 6);
                        }
                    }

                    $this->_view->partialPayment = true;
                    if (empty($partialPercent)) {
                        $this->_view->partialPayment = false;
                    }

                    $this->_view->partialPercent = $partialPercent;
                }


                $this->_view->totalTax = round($cart->getTotalTax(), 2);
                $this->_view->shipping = round($cart->getShippingPrice(), 2);
                $this->_view->totalAmount = round($cart->getTotal(), 2);
                $this->_view->cartTotal = $cart->getTotal();
                $this->_view->currency = Models_Mapper_ShoppingConfig::getInstance()->getConfigParam('currency');
                $shippingAddressKey = $cart->getShippingAddressId();
                if ($shippingAddressKey != null) {
                    $this->_view->allowShipping = true;
                }
                $this->_view->countryList = $this->_prepareCountryList();
                if (Models_Mapper_ShoppingConfig::getInstance()->getConfigParam('country') != '') {
                    $predefinedCountry = Models_Mapper_ShoppingConfig::getInstance()->getConfigParam('country');
                } else {
                    $predefinedCountry = 'US';
                }

                $phonecountrycodes = Tools_System_Tools::getFullCountryPhoneCodesList(true, array(), true);
                $phonecountrycode = $predefinedCountry;

                $listMasksMapper = Application_Model_Mappers_MasksListMapper::getInstance();
                $this->_view->phonecountrycodes = $phonecountrycodes;
                $this->_view->desktopMasks = $listMasksMapper->getListOfMasksByType(Application_Model_Models_MaskList::MASK_TYPE_DESKTOP);

                $billingAddressKey = $cart->getBillingAddressId();
                if ($billingAddressKey != null) {
                    $paymentParams = Tools_ShoppingCart::getAddressById($billingAddressKey);
                    if (isset($paymentParams['state'])) {
                        $state = Tools_Geo::getStateById($paymentParams['state']);
                        $paymentParams['state'] = $state['id'];
                        $this->_view->state = $paymentParams['state'];
                    }

                    $phonecountrycode = $paymentParams['phonecountrycode'];

                    $this->_view->firstName = $paymentParams['firstname'];
                    $this->_view->lastName = $paymentParams['lastname'];
                    $this->_view->address1 = $paymentParams['address1'];
                    $this->_view->address2 = $paymentParams['address2'];
                    $this->_view->billingEmail = $paymentParams['email'];
                    $this->_view->city = $paymentParams['city'];
                    $this->_view->zip = $paymentParams['zip'];
                    $this->_view->phone = $paymentParams['phone'];
                    $predefinedCountry = $paymentParams['country'];

                }

                $this->_view->phonecountrycode = $phonecountrycode;

                $listOfStates = $this->_stateCheck($predefinedCountry);
                if ($listOfStates['error'] == '0') {
                    $this->_view->listOfState = $listOfStates[0];
                }

                $this->_view->predefinedCountry = $predefinedCountry;

                $amountToPay = $this->_view->totalAmount;
                if ($paymentType === Quote_Models_Model_Quote::PAYMENT_TYPE_PARTIAL_PAYMENT) {
                    $partialPaidAmount = $cart->getPartialPaidAmount();
                    $partialPaymentType = $cart->getPartialType();
                    if (!empty($partialPaidAmount) && $partialPaidAmount !== '0.00' && !empty((int)$partialPaidAmount)) {
                        if ($partialPaymentType === Models_Model_CartSession::CART_PARTIAL_PAYMENT_TYPE_AMOUNT) {
                            $amountToPay = round($cart->getTotal() - $cart->getPartialPercentage(), 2);
                        } else {
                            $amountToPay = round($cart->getTotal() - round(($cart->getTotal() * $cart->getPartialPercentage()) / 100, 2), 2);
                        }
                    } else {
                        if ($partialPaymentType === Models_Model_CartSession::CART_PARTIAL_PAYMENT_TYPE_AMOUNT) {
                            $amountToPay = round($cart->getPartialPercentage(), 2);
                        } else {
                            $amountToPay = round(($cart->getTotal() * $cart->getPartialPercentage()) / 100, 2);
                        }
                    }
                }

                $currency = Zend_Registry::get('Zend_Currency');

                $this->_view->amountToPay = $currency->toCurrency($amountToPay);

                return $this->_view->render('creditcard-card-quote.phtml');
            }

        }


        return '';
    }

    /**
     * Prepare states list
     *
     * @param string $country
     * @return array
     */
    private function _stateCheck($country)
    {
        $pairs = true;
        $stateData = Tools_Geo::getState($country, $pairs);
        if (!empty ($stateData)) {
            $states = '';
            foreach ($stateData as $short => $state) {
                $states .= '<option value="' . $short . '">' . $state . '</option>';
            }
            return array('error' => '0', $states);
        } else {
            return array('error' => '1');
        }
    }

    public function _makeOptionGoogleapplepayquotebutton()
    {

        if (Tools_Security_Acl::isAllowed(Shopping::RESOURCE_STORE_MANAGEMENT)) {
            return '';
        }

        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();
        if(!empty($stripepayOAuth)) {
            $stripeConfig = Tools_StripeOAuthTools::connectionStatusMojo();
            $publishKey = $stripeConfig['stripe_publishable_key'];
            $accountId = $stripeConfig['stripe_client_id'];
            $disableLink = Stripepay_Models_Mappers_StripepayConfigMapper::getInstance()->getConfigParam('disableLink');

            if(!empty($disableLink)) {
                $disableLink = 1;
            }
        } else {
            $stripeConfig = $this->_getStripeConfig();
            $publishKey = $stripeConfig['publishKey'];
            $disableLink = $stripeConfig['disableLink'];
        }

        if (!empty($publishKey)) {
            $this->_view->translator = $this->_translator;

            $pageHelper = Zend_Controller_Action_HelperBroker::getStaticHelper('page');
            $front = Zend_Controller_Front::getInstance();
            $productMapper = Models_Mapper_ProductMapper::getInstance();
            $quote = Quote_Models_Mapper_QuoteMapper::getInstance()->find(
                $pageHelper->clean($front->getRequest()->getParams('page'))
            );
            if (!$quote instanceof Quote_Models_Model_Quote) {
                throw new Exceptions_SeotoasterPluginException('Quote not found');
            }


            $cartSessionMapper = Models_Mapper_CartSessionMapper::getInstance();
            $cart = $cartSessionMapper->find($quote->getCartId());

            if (!$cart instanceof Models_Model_CartSession) {
                throw new Exceptions_SeotoasterException('Cart not found');
            }
            if ($quote->getStatus() === Quote_Models_Model_Quote::STATUS_LOST) {
                return null;
            }

            $recalculationFlags = $this->_setRecalculationFlags();
            if (!empty($quote->getCreatorId())) {
                $recalculationFlags = $this->_setRecalculationFlags($quote->getCreatorId());
            }
            if (!empty($quote->getEditorId())) {
                $recalculationFlags = $this->_setRecalculationFlags($quote->getEditorId());
            }
            $cartContent = $cart->getCartContent();
            $discount = $cart->getDiscount();
            $discountRate = $cart->getDiscountTaxRate();

            if (!$cartContent) {
                return null;
            }

            $status = $cart->getStatus();

            if ($status != Models_Model_CartSession::CART_STATUS_COMPLETED && $status != Models_Model_CartSession::CART_STATUS_DELIVERED && $status != Models_Model_CartSession::CART_STATUS_SHIPPED) {
                if($status === Models_Model_CartSession::CART_STATUS_PENDING && $cart->getGateway() !== 'Quote'){
                    return '';
                }

                $cartSession = Tools_ShoppingCart::getInstance();
                $cartSession->setContent(array());
                $cartSession->save();
                $cartSession->setCartId($quote->getCartId());
                $cartSession->setShippingAddressKey($cart->getShippingAddressId());
                $cartSession->setBillingAddressKey($cart->getBillingAddressId());
                $notFreebiesInCart = array();
                $freebiesInCart = array();
                $productsFreebiesRelation = array();

                foreach ($cartContent as $key => $product) {
                    if ($product['freebies'] === '1') {
                        $freebiesInCart[$product['product_id']] = $product['product_id'];
                    } else {
                        $notFreebiesInCart[$product['product_id']] = $product['product_id'];
                    }
                }

                if (!empty($freebiesInCart)) {
                    $where = $productMapper->getDbTable()->getAdapter()->quoteInto(
                        'sphp.freebies_id IN (?)',
                        $freebiesInCart
                    );
                    $where .= ' AND ' . $productMapper->getDbTable()->getAdapter()->quoteInto(
                            'sphp.product_id IN (?)',
                            $notFreebiesInCart
                        );
                    $select = $productMapper->getDbTable()->getAdapter()->select()
                        ->from(
                            array('spfs' => 'shopping_product_freebies_settings'),
                            array(
                                'freebiesGroupKey' => new Zend_Db_Expr("CONCAT(sphp.freebies_id, '_', sphp.product_id)"),
                                'price_value'
                            )
                        )
                        ->joinleft(
                            array('sphp' => 'shopping_product_has_freebies'),
                            'spfs.prod_id = sphp.product_id'
                        )
                        ->where($where);
                    $productFreebiesSettings = $productMapper->getDbTable()->getAdapter()->fetchAssoc($select);
                }

                if (!empty($productFreebiesSettings)) {
                    foreach ($productFreebiesSettings as $prodInfo) {
                        if (array_key_exists($prodInfo['freebies_id'], $freebiesInCart)) {
                            if (isset($productsFreebiesRelation[$prodInfo['freebies_id']])) {
                                $productsFreebiesRelation[$prodInfo['freebies_id']][$prodInfo['product_id']] = $prodInfo['product_id'];
                            } else {
                                $productsFreebiesRelation[$prodInfo['freebies_id']] = array($prodInfo['product_id'] => $prodInfo['product_id']);
                            }
                        }
                    }
                }

                foreach ($cartContent as $key => $product) {
                    $productObject = $productMapper->find($product['product_id']);
                    if ($productObject instanceof Models_Model_Product) {
                        if ($product['freebies'] === '1' && !empty($productsFreebiesRelation)) {
                            foreach ($productsFreebiesRelation[$product['product_id']] as $realProductId) {
                                $itemKey = $this->_generateStorageKey(
                                    $productObject,
                                    array(0 => 'freebies_' . $realProductId)
                                );
                                if (!$cartSession->findBySid($itemKey)) {
                                    $productObject->setFreebies(1);
                                    $cartSession->add(
                                        $productObject,
                                        array(0 => 'freebies_' . $realProductId),
                                        $product['qty'], true, $recalculationFlags['skipOptionRecalculation'], $recalculationFlags['skipGroupPriceRecalculation']
                                    );
                                }
                            }
                        } else {
                            $options = array();
                            if (is_array($product['options'])) {
                                $options = $this->_parseProductOptions($product['options']);
                            }
                            $productObject->setPrice($product['price']);
                            $productObject->setOriginalPrice($product['original_price']);
                            $productObject->setCurrentPrice(floatval($productObject->getPrice()));
                            $cartSession->add($productObject, $options, $product['qty'], true, $recalculationFlags['skipOptionRecalculation'], $recalculationFlags['skipGroupPriceRecalculation']);
                        }
                    }
                }

                $cartSession->setDiscount($discount);
                $cartSession->setShippingData(array('price' => $cart->getShippingPrice()));
                $cartSession->setDiscountTaxRate($discountRate);
                $cartSession->calculate(true , $recalculationFlags['skipGroupPriceRecalculation']);
                $cartSession->save();

                $paymentType = $quote->getPaymentType();
                $partialPaymentType = $cart->getPartialType();

                if (Models_Mapper_ShoppingConfig::getInstance()->getConfigParam('country') != '') {
                    $predefinedCountry = Models_Mapper_ShoppingConfig::getInstance()->getConfigParam('country');
                } else {
                    $predefinedCountry = 'US';
                }

                if(!in_array($predefinedCountry, $this->_googleApplePayAllowedCountires)) {
                    throw new Exceptions_SeotoasterException('Your store country is not supported by Stripepay');
                }

                //$this->_view->totalAmount = round($cart->getTotal(), 2);
                $amountToPay = round($cart->getTotal(), 2);
                if ($paymentType === Quote_Models_Model_Quote::PAYMENT_TYPE_PARTIAL_PAYMENT) {
                    $partialPaidAmount = $cart->getPartialPaidAmount();
                    if (!empty($partialPaidAmount) && $partialPaidAmount !== '0.00' && !empty((int) $partialPaidAmount)) {
                        if ($partialPaymentType === Models_Model_CartSession::CART_PARTIAL_PAYMENT_TYPE_AMOUNT) {
                            $amountToPay = round($cart->getTotal() - $cart->getPartialPercentage(),2);
                        } else {
                            $amountToPay = round($cart->getTotal() - round(($cart->getTotal() * $cart->getPartialPercentage()) / 100,
                                    2), 2);
                        }
                    } else {
                        if ($partialPaymentType === Models_Model_CartSession::CART_PARTIAL_PAYMENT_TYPE_AMOUNT) {
                            $amountToPay = round($cart->getPartialPercentage(), 2);
                        } else {
                            $amountToPay = round(($cart->getTotal() * $cart->getPartialPercentage()) / 100, 2);
                        }
                    }
                }

                $totalAmount = round($cart->getTotal(), 2) * 100;
                $amountToPay = $amountToPay * 100;

                $this->_view->amountToPay = $amountToPay;
                $this->_view->currency = Models_Mapper_ShoppingConfig::getInstance()->getConfigParam('currency');
                $this->_view->country = $predefinedCountry;

                $cartContent = Tools_ShoppingCart::getInstance();
                $productList = array_values($cartContent->getContent());

                if (!empty ($productList)) {
                    $cartParams = $cartContent->calculate();
                    $cartId = $cartContent->getCartId();
                    $this->_view->totalAmount = $totalAmount;
                    $this->_view->publishKey = $publishKey;

                    if (!empty($productList) && (!empty($stripeConfig['privateKey'] || !empty($stripepayOAuth)))) {
                        $cartSession = $cartSessionMapper->find($cartId);

                        if ($cartSession instanceof Models_Model_CartSession) {
                            $dataRequest = array(
                                'amount' => $amountToPay,
                                'currency' => strtolower($this->_view->currency),//'usd',
                                'payment_method_types' => ['card'],
                            );

                            $this->_view->accountId = '';
                            if(!empty($stripepayOAuth)) {
                                $responcePaymentIntent = Tools_StripeOAuthTools::createPaymentIntentMojo($dataRequest);

                                if(!empty($responcePaymentIntent)) {
                                    $responcePaymentIntentData = json_decode($responcePaymentIntent, true);

                                    if(!empty($responcePaymentIntentData)) {
                                        $intent = (object) $responcePaymentIntentData;
                                        $this->_view->accountId = $accountId;
                                    }
                                }
                            } else {
                                Stripe::setApiKey($stripeConfig['privateKey']);
                                $intent = PaymentIntent::create($dataRequest);
                            }

                            if(isset($intent->client_secret)) {
                                try {
                                    $liveModeParam = $intent->livemode;
                                    $liveMode = 0;
                                    if($liveModeParam) {
                                        $liveMode = 1;
                                    }
                                } catch (Exception $e) {
                                    error_log($e->getMessage());
                                }

                                $lineItems = [];
                                foreach ($productList as $product) {
                                    $lineItem = [];
                                    $lineItem['label'] = $product['name'];
                                    $lineItem['amount'] = round($product['taxPrice']* $product['qty'], 2) * 100;
                                    $lineItem['amount'] = intval($lineItem['amount']);
                                    $lineItems[] = $lineItem;
                                }

                                $shippingService = $cartSession->getShippingService();
                                $shippingType = $cartSession->getShippingType();
                                $shippingPrice = $cartSession->getShippingPrice();

                                $shippingLabel = '';
                                $shippingDetail = '';
                                $shippingAmount = 0;
                                if(!empty($shippingService)) {
                                    $shippingLabel = $shippingService;

                                    if(!empty($shippingType)) {
                                        $shippingDetail = $shippingType;
                                    }

                                    $shippingAmount = round($shippingPrice, 2) * 100;
                                }

                                $this->_view->clientSecret = $intent->client_secret;
                                $this->_view->disableLink = $disableLink;
                                $this->_view->liveMode = $liveMode;
                                $this->_view->country = $predefinedCountry;
                                $this->_view->currency = strtolower($this->_view->currency);
                                $this->_view->shippingLabel = $shippingLabel;
                                $this->_view->shippingDetail = $shippingDetail;
                                $this->_view->shippingAmount = $shippingAmount;
                                $this->_view->lineItems = json_encode($lineItems);

                                return $this->_view->render('googleapplePayButtonQuote.phtml');
                            }
                        }
                    }
                }
            }
        } else {
            return $this->_translator->translate('Stripe not configured!');
        }
    }
    public function _makeOptionButton()
    {
        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();
        if(!empty($stripepayOAuth)) {
            $stripeConfig = Tools_StripeOAuthTools::connectionStatusMojo();
            $publishKey = $stripeConfig['stripe_publishable_key'];
            $disableLink = Stripepay_Models_Mappers_StripepayConfigMapper::getInstance()->getConfigParam('disableLink');

            if(!empty($disableLink)) {
                $disableLink = 1;
            }
            $this->_view->accountId = $stripeConfig['stripe_client_id'];
        } else {
            $stripeConfig = $this->_getStripeConfig();
            $publishKey = $stripeConfig['publishKey'];
            $disableLink = $stripeConfig['disableLink'];
            $this->_view->accountId = '';
        }

        $this->_view->translator = $this->_translator;
        if (empty($publishKey)) {
            return $this->_translator->translate('Stripe not configured!');
        }
        $cartContent = Tools_ShoppingCart::getInstance();
        $shoppingConfigMapper = Models_Mapper_ShoppingConfig::getInstance();
        $productList = array_values($cartContent->getContent());

        if (!empty ($productList)) {
            $cartParams = $cartContent->calculate();
            $cartId = $cartContent->getCartId();
            $this->_view->totalTax = round($cartParams['totalTax'], 2);
            $this->_view->shipping = round($cartParams['shipping'], 2);
            $this->_view->totalAmount = round($cartParams['total'], 2);
            $this->_view->discount = round($cartParams['discount'], 2);
            $this->_view->cartId = $cartId;
            $this->_view->publishKey = $publishKey;

            if(in_array('google-apple-pay', $this->_options)) {
                $cartSessionMapper = Models_Mapper_CartSessionMapper::getInstance();
                $cartSession = $cartSessionMapper->find($cartId);

                if ($cartSession instanceof Models_Model_CartSession) {
                    $country = $shoppingConfigMapper->getConfigParam('country');
                    if ($country != '') {
                        $predefinedCountry = $country;
                    } else {
                        $predefinedCountry = 'US';
                    }

                    if(!in_array($predefinedCountry, $this->_googleApplePayAllowedCountires)) {
                        throw new Exceptions_SeotoasterException('Your store country is not supported by Stripepay');
                    }

                    $gateway = $cartSession->getGateway();
                    $cartStatus = $cartSession->getStatus();
                    if (!in_array($cartStatus, $this->_allowedStatusesCheckout) && $gateway == 'Stripepay') {
                        $this->_view->cartStatusError = true;
                        unset($this->_sessionHelper->storeCartSessionKey);
                        unset($this->_sessionHelper->storeCartSessionConversionKey);
                        $cartContent->clean();
                    }

                    $productList = array_values($cartContent->getContent());
                    if (!empty($productList) && (!empty($stripeConfig['privateKey'] || !empty($stripepayOAuth)))) {

                        $this->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_PROCESSING);

                        $shippingAddressKey = Tools_ShoppingCart::getInstance()->getAddressKey(Models_Model_Customer::ADDRESS_TYPE_SHIPPING);
                        if ($shippingAddressKey != null) {
                            $customerShipping = Tools_ShoppingCart::getAddressById($shippingAddressKey);
                            $state = Tools_Geo::getStateById($customerShipping['state']);
                            $customerShipping['state'] = $state['state'];
                        }

                        $billingAddressArray = array();
                        $billingAddressArray['firstname'] = $customerShipping['firstname'];
                        $billingAddressArray['lastname'] = $customerShipping['lastname'];
                        $billingAddressArray['company'] = '';
                        $billingAddressArray['email'] = $customerShipping['email'];
                        $billingAddressArray['address1'] = $customerShipping['address1'];
                        $billingAddressArray['address2'] = '';
                        $billingAddressArray['country'] = $customerShipping['country'];
                        $billingAddressArray['city'] = $customerShipping['city'];
                        $billingAddressArray['state'] = $customerShipping['state'];
                        $billingAddressArray['zip'] = $customerShipping['zip'];
                        $billingAddressArray['phone'] = isset($customerShipping['phone']) ? $customerShipping['phone'] : '';
                        $billingAddressArray['phone_country_code_value'] = isset($customerShipping['phone_country_code_value']) ? $customerShipping['phone_country_code_value'] : '';
                        $billingAddressArray['phonecountrycode'] = isset($customerShipping['phonecountrycode']) ? $customerShipping['phonecountrycode'] : '';

                        $addressId = $this->_addAddress($cartSession->getUserId(), $billingAddressArray, Models_Model_Customer::ADDRESS_TYPE_BILLING);
                        $cartSession->setBillingAddressId($addressId);
                        $cartSessionMapper->save($cartSession);

                        $cartParams = $cartContent->calculate();

                        /*
                        //@todo use an activate Apple Pay site domain https://stripe.com/docs/stripe-js/elements/payment-request-button#verifying-your-domain-with-apple-pay
                        if(!$stripeConfig['useSandBox']) {
                            Stripe::setApiKey($stripeConfig['publishKey']); //sk_live_••••••••••••••••••••••••

                            $domain = Tools_StripeRegular::getDomainFromUrl($this->_websiteUrl);
                            if(!empty($domain)) {
                                ApplePayDomain::create([
                                    'domain_name' => $domain,//'example.com',
                                ]);
                            }

                        } else {
                            Stripe::setApiKey($stripeConfig['privateKey']);
                        }*/

                        $totalAmount = round($cartParams['total'], 2) * 100;
                        $totalAmountOriginal = round($cartParams['total'], 2);

                        $dataRequest = array(
                            'amount' => $totalAmount,
                            'currency' => strtolower($shoppingConfigMapper->getConfigParam('currency')),//'usd',
                            'payment_method_types' => ['card']
                        );
                        if(!empty($stripepayOAuth)) {
                            $responcePaymentIntent = Tools_StripeOAuthTools::createPaymentIntentMojo($dataRequest);

                            if(!empty($responcePaymentIntent)) {
                                $responcePaymentIntentData = json_decode($responcePaymentIntent, true);

                                if(!empty($responcePaymentIntentData)) {
                                    $intent = (object) $responcePaymentIntentData;
                                }
                            }
                        } else {
                            Stripe::setApiKey($stripeConfig['privateKey']);
                            $intent = PaymentIntent::create($dataRequest);
                        }

                        if(isset($intent->client_secret)) {
                            try {
                                $liveModeParam = $intent->livemode;
                                $liveMode = 0;
                                if($liveModeParam) {
                                    $liveMode = 1;
                                }

                                $stripeTransactionMapper = Stripepay_Models_Mappers_StripepayTransactionMapper::getInstance();

                                $stripeTransactionModel = $stripeTransactionMapper->selectTransactionsByCartId($cartId, true);

                                if(!$stripeTransactionModel instanceof Stripepay_Models_Models_StripepayTransactionModel) {
                                    $stripeTransactionModel = new Stripepay_Models_Models_StripepayTransactionModel();
                                }

                                $stripeTransactionModel->setLiveMode($liveMode);
                                $stripeTransactionModel->setCheckoutSessionId('');
                                $stripeTransactionModel->setStripeCustomerId('');
                                //$stripeTransactionModel->setPaymentIntent($intent->id);
                                $stripeTransactionModel->setStatus(self::TRANSACTION_STATUS_PROCESSING);
                                $stripeTransactionModel->setAmount($cartContent->getTotal());
                                $stripeTransactionModel->setTotalTax($cartContent->getTotalTax());
                                $stripeTransactionModel->setCartId($cartContent->getCartId());
                                $stripeTransactionModel->setOrderShippingAddress($customerShipping ? json_encode($customerShipping) : '');
                                $stripeTransactionModel->setOrderDateFull(Tools_System_Tools::convertDateFromTimezone('now'));
                                $stripeTransactionModel->setCurrency($shoppingConfigMapper->getConfigParam('currency'));
                                $stripeTransactionModel->setTransactionType(self::TRANSACTION_STATUS_PROCESSING);
                                $stripeTransactionModel->setPaymentType(self::PAYMENT_TYPE_BUTTON);
                                //$stripeTransactionModel->setIntentClientSecret($intent->client_secret);
                                $stripeTransactionModel->setFirstName($customerShipping['firstname'] ? $customerShipping['firstname'] : '');
                                $stripeTransactionModel->setLastName($customerShipping['lastname'] ? $customerShipping['lastname'] : '');
                                $stripeTransactionModel->setEmail($customerShipping['email'] ? $customerShipping['email'] : '');
                                $stripeTransactionModel->setReason('');

                                $stripeTransactionMapper->save($stripeTransactionModel);
                            } catch (Exception $e) {
                                error_log($e->getMessage());
                            }

                            $lineItems = [];
                            foreach ($productList as $product) {
                                $lineItem = [];
                                $lineItem['label'] = $product['name'];
                                $lineItem['amount'] = round($product['taxPrice']* $product['qty'], 2) * 100;
                                $lineItem['amount'] = intval($lineItem['amount']);
                                $lineItems[] = $lineItem;
                            }

                            $shippingService = $cartSession->getShippingService();
                            $shippingType = $cartSession->getShippingType();
                            $shippingPrice = $cartSession->getShippingPrice();

                            $shippingLabel = '';
                            $shippingDetail = '';
                            $shippingAmount = 0;
                            if(!empty($shippingService)) {
                                $shippingLabel = $shippingService;

                                if(!empty($shippingType)) {
                                    $shippingDetail = $shippingType;
                                }

                                $shippingAmount = round($shippingPrice, 2) * 100;
                            }

                            $this->_view->clientSecret = $intent->client_secret;
                            $this->_view->disableLink = $disableLink;
                            $this->_view->totalAmount = $totalAmount;
                            $this->_view->totalAmountOriginal = $totalAmountOriginal;
                            $this->_view->country = $predefinedCountry;
                            $this->_view->currency = strtolower($shoppingConfigMapper->getConfigParam('currency'));
                            $this->_view->transactionId = $stripeTransactionModel->getId();
                            $this->_view->shippingLabel = $shippingLabel;
                            $this->_view->shippingDetail = $shippingDetail;
                            $this->_view->shippingAmount = $shippingAmount;
                            $this->_view->lineItems = json_encode($lineItems);

                            return $this->_view->render('googleapplePayButton.phtml');
                        }
                    }

                }
            } else {
                return $this->_view->render('button.phtml');
            }

        }
    }

    public function _makeOptionCreditcardquoterecurrent()
    {
        if (Tools_Security_Acl::isAllowed(Shopping::RESOURCE_STORE_MANAGEMENT)) {
            return '';
        }

        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();
        if(!empty($stripepayOAuth)) {
            $stripeConfig = Tools_StripeOAuthTools::connectionStatusMojo();
            $publishKey = $stripeConfig['stripe_publishable_key'];
            $accountId = $stripeConfig['stripe_client_id'];
        } else {
            $stripeConfig = $this->_getStripeConfig();
            $publishKey = $stripeConfig['publishKey'];
            $privateKey = $stripeConfig['privateKey'];
        }

        if(!empty($stripeConfig)) {
            $this->_view->translator = $this->_translator;
            $shoppingConfigMapper = Models_Mapper_ShoppingConfig::getInstance();
            $shoppingConfig = $shoppingConfigMapper->getConfigParams();

            if (empty($publishKey)) {
                return $this->_translator->translate('Stripe not configured!');
            }

            $pageHelper = Zend_Controller_Action_HelperBroker::getStaticHelper('page');
            $front = Zend_Controller_Front::getInstance();
            $productMapper = Models_Mapper_ProductMapper::getInstance();
            $quote = Quote_Models_Mapper_QuoteMapper::getInstance()->find(
                $pageHelper->clean($front->getRequest()->getParams('page'))
            );
            if (!$quote instanceof Quote_Models_Model_Quote) {
                throw new Exceptions_SeotoasterPluginException('Quote not found');
            }
            $cart = Models_Mapper_CartSessionMapper::getInstance()->find($quote->getCartId());
            if (!$cart instanceof Models_Model_CartSession) {
                throw new Exceptions_SeotoasterException('Cart not found');
            }
            if ($quote->getStatus() === Quote_Models_Model_Quote::STATUS_LOST) {
                return null;
            }
            $recalculationFlags = $this->_setRecalculationFlags();
            if (!empty($quote->getCreatorId())) {
                $recalculationFlags = $this->_setRecalculationFlags($quote->getCreatorId());
            }
            if (!empty($quote->getEditorId())) {
                $recalculationFlags = $this->_setRecalculationFlags($quote->getEditorId());
            }

            $paymentType = $quote->getPaymentType();
            if ($paymentType === Quote_Models_Model_Quote::PAYMENT_TYPE_PARTIAL_PAYMENT) {
               return '';
            }

            $cartContent = $cart->getCartContent();
            $discount = $cart->getDiscount();
            $discountRate = $cart->getDiscountTaxRate();
            if (!$cartContent) {
                return null;
            }
            $status = $cart->getStatus();

            if ($status != Models_Model_CartSession::CART_STATUS_COMPLETED && $status != Models_Model_CartSession::CART_STATUS_DELIVERED && $status != Models_Model_CartSession::CART_STATUS_SHIPPED) {
                if($status === Models_Model_CartSession::CART_STATUS_PENDING && $cart->getGateway() !== 'Quote'){
                    return '';
                }

                if (in_array('period', array_values($this->_options))) {
                    $periods = $this->_processRecurringPeriods();
                    if (empty($periods)) {
                        return $this->_translator->translate('Wrong periods provided.');
                    }
                }

                $intent = Tools_RecurringPaymentStripepay::createSetupIntent($privateKey);

                $this->_view->accountId = '';
                if(!empty($intent['intent'])) {
                    $intentData = json_decode($intent['intent'], true);

                    if(!empty($intentData)) {
                        $intent = (object) $intentData;
                        $this->_view->accountId = $accountId;
                    }
                }

                if (!is_object($intent)) {
                    return $this->_translator->translate('Stripe intent cannot be created!');
                }
                $this->_view->intent = $intent;
                $this->_view->publishKey = $publishKey;
                $this->_view->periods = $periods;
                $this->_view->organizationCountry = $shoppingConfig['country'];

                $cartSession = Tools_ShoppingCart::getInstance();
                $cartSession->setContent(array());
                $cartSession->save();
                $cartSession->setCartId($quote->getCartId());
                $cartSession->setShippingAddressKey($cart->getShippingAddressId());
                $cartSession->setBillingAddressKey($cart->getBillingAddressId());
                $notFreebiesInCart = array();
                $freebiesInCart = array();
                $productsFreebiesRelation = array();

                foreach ($cartContent as $key => $product) {
                    if ($product['freebies'] === '1') {
                        $freebiesInCart[$product['product_id']] = $product['product_id'];
                    } else {
                        $notFreebiesInCart[$product['product_id']] = $product['product_id'];
                    }
                }

                if (!empty($freebiesInCart)) {
                    $where = $productMapper->getDbTable()->getAdapter()->quoteInto(
                        'sphp.freebies_id IN (?)',
                        $freebiesInCart
                    );
                    $where .= ' AND ' . $productMapper->getDbTable()->getAdapter()->quoteInto(
                            'sphp.product_id IN (?)',
                            $notFreebiesInCart
                        );
                    $select = $productMapper->getDbTable()->getAdapter()->select()
                        ->from(
                            array('spfs' => 'shopping_product_freebies_settings'),
                            array(
                                'freebiesGroupKey' => new Zend_Db_Expr("CONCAT(sphp.freebies_id, '_', sphp.product_id)"),
                                'price_value'
                            )
                        )
                        ->joinleft(
                            array('sphp' => 'shopping_product_has_freebies'),
                            'spfs.prod_id = sphp.product_id'
                        )
                        ->where($where);
                    $productFreebiesSettings = $productMapper->getDbTable()->getAdapter()->fetchAssoc($select);
                }

                if (!empty($productFreebiesSettings)) {
                    foreach ($productFreebiesSettings as $prodInfo) {
                        if (array_key_exists($prodInfo['freebies_id'], $freebiesInCart)) {
                            if (isset($productsFreebiesRelation[$prodInfo['freebies_id']])) {
                                $productsFreebiesRelation[$prodInfo['freebies_id']][$prodInfo['product_id']] = $prodInfo['product_id'];
                            } else {
                                $productsFreebiesRelation[$prodInfo['freebies_id']] = array($prodInfo['product_id'] => $prodInfo['product_id']);
                            }
                        }
                    }
                }

                foreach ($cartContent as $key => $product) {
                    $productObject = $productMapper->find($product['product_id']);
                    if ($productObject instanceof Models_Model_Product) {
                        if ($product['freebies'] === '1' && !empty($productsFreebiesRelation)) {
                            foreach ($productsFreebiesRelation[$product['product_id']] as $realProductId) {
                                $itemKey = $this->_generateStorageKey(
                                    $productObject,
                                    array(0 => 'freebies_' . $realProductId)
                                );
                                if (!$cartSession->findBySid($itemKey)) {
                                    $productObject->setFreebies(1);
                                    $cartSession->add(
                                        $productObject,
                                        array(0 => 'freebies_' . $realProductId),
                                        $product['qty'], true, $recalculationFlags['skipOptionRecalculation'], $recalculationFlags['skipGroupPriceRecalculation']
                                    );
                                }
                            }
                        } else {
                            $options = array();
                            if (is_array($product['options'])) {
                                $options = $this->_parseProductOptions($product['options']);
                            }
                            $productObject->setPrice($product['price']);
                            $productObject->setOriginalPrice($product['original_price']);
                            $productObject->setCurrentPrice(floatval($productObject->getPrice()));
                            $cartSession->add($productObject, $options, $product['qty'], true, $recalculationFlags['skipOptionRecalculation'], $recalculationFlags['skipGroupPriceRecalculation']);
                        }
                    }
                }

                $cartSession->setDiscount($discount);
                $cartSession->setShippingData(array('price' => $cart->getShippingPrice()));
                $cartSession->setDiscountTaxRate($discountRate);
                $cartSession->calculate(true , $recalculationFlags['skipGroupPriceRecalculation']);
                $cartSession->save();

                $this->_view->totalTax = round($cart->getTotalTax(), 2);
                $this->_view->shipping = round($cart->getShippingPrice(), 2);
                $this->_view->totalAmount = round($cart->getTotal(), 2);
                $this->_view->cartTotal = $cart->getTotal();
                $this->_view->currency = Models_Mapper_ShoppingConfig::getInstance()->getConfigParam('currency');
                $this->_view->countryList = $this->_prepareCountryList();
                if (Models_Mapper_ShoppingConfig::getInstance()->getConfigParam('country') != '') {
                    $predefinedCountry = Models_Mapper_ShoppingConfig::getInstance()->getConfigParam('country');
                } else {
                    $predefinedCountry = 'US';
                }

                $phonecountrycodes = Tools_System_Tools::getFullCountryPhoneCodesList(true, array(), true);
                $phonecountrycode = $predefinedCountry;

                $listMasksMapper = Application_Model_Mappers_MasksListMapper::getInstance();
                $this->_view->phonecountrycodes = $phonecountrycodes;
                $this->_view->desktopMasks = $listMasksMapper->getListOfMasksByType(Application_Model_Models_MaskList::MASK_TYPE_DESKTOP);
                $this->_view->phone = '';

                $billingAddressKey = $cart->getBillingAddressId();
                if ($billingAddressKey != null) {
                    $paymentParams = Tools_ShoppingCart::getAddressById($billingAddressKey);
                    if (isset($paymentParams['state'])) {
                        $state = Tools_Geo::getStateById($paymentParams['state']);
                        $paymentParams['state'] = $state['id'];
                        $this->_view->state = $paymentParams['state'];
                    }

                    $phonecountrycode = $paymentParams['phonecountrycode'];

                    $this->_view->firstName = $paymentParams['firstname'];
                    $this->_view->lastName = $paymentParams['lastname'];
                    $this->_view->address1 = $paymentParams['address1'];
                    $this->_view->address2 = $paymentParams['address2'];
                    $this->_view->billingEmail = $paymentParams['email'];
                    $this->_view->city = $paymentParams['city'];
                    $this->_view->zip = $paymentParams['zip'];
                    $this->_view->phone = $paymentParams['phone'];
                    $predefinedCountry = $paymentParams['country'];
                }

                $this->_view->phonecountrycode = $phonecountrycode;

                $listOfStates = $this->prepareStateList($predefinedCountry);
                if ($listOfStates['error'] == '0') {
                    $this->_view->listOfState = $listOfStates[0];
                }

                $this->_view->predefinedCountry = $predefinedCountry;

                return $this->_view->render('stripepay-credit-card-quote.phtml');
            }
        }

        return '';

    }

    public function _makeOptionProduct()
    {
        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();
        if(!empty($stripepayOAuth)) {
            $stripeConfig = Tools_StripeOAuthTools::connectionStatusMojo();
            $publishKey = $stripeConfig['stripe_publishable_key'];
            $accountId = $stripeConfig['stripe_client_id'];
        } else {
            $stripeConfig = $this->_getStripeConfig();
            $publishKey = $stripeConfig['publishKey'];
            $privateKey = $stripeConfig['privateKey'];
        }

        $this->_view->translator = $this->_translator;
        $shoppingConfigMapper = Models_Mapper_ShoppingConfig::getInstance();
        $shoppingConfig = $shoppingConfigMapper->getConfigParams();
        $session = Zend_Controller_Action_HelperBroker::getStaticHelper('session');
        if (empty($publishKey)) {
            return $this->_translator->translate('Stripe not configured!');
        }
        if (in_array('period', array_values($this->_options))) {
            $periods = $this->_processRecurringPeriods();
            if (empty($periods)) {
                return $this->_translator->translate('Wrong periods provided.');
            }
        }

        $currentLoggedUser = $session->getCurrentUser();
        if ($currentLoggedUser instanceof Application_Model_Models_User && $currentLoggedUser->getId()) {
            $userFullName = explode(' ', $currentLoggedUser->getFullName());
            $this->_view->firstName = is_array($userFullName) ? $userFullName[0] : '';
            $this->_view->lastName = isset($userFullName[1]) ? $userFullName[1] : '';
            $this->_view->billingEmail = $currentLoggedUser->getEmail();
            $this->_view->isUserLoggedIn = true;
        } else {
            $this->_view->firstName = '';
            $this->_view->lastName = '';
            $this->_view->billingEmail = '';
            $this->_view->address1 = '';
            $this->_view->city = '';
            $this->_view->zip = '';
            $this->_view->isUserLoggedIn = false;
        }
        $this->_view->countryList = $this->_prepareCountryList();
        $country = $shoppingConfigMapper->getConfigParam('country');
        if ($country != '') {
            $predefinedCountry = $country;
        } else {
            $predefinedCountry = 'US';
        }
        $phonecountrycodes = Tools_System_Tools::getFullCountryPhoneCodesList(true, array(), true);
        $this->_view->phonecountrycode = $predefinedCountry;
        $listMasksMapper = Application_Model_Mappers_MasksListMapper::getInstance();
        $this->_view->phonecountrycodes = $phonecountrycodes;
        $this->_view->desktopMasks = $listMasksMapper->getListOfMasksByType(Application_Model_Models_MaskList::MASK_TYPE_DESKTOP);
        $this->_view->phone = '';
        $intent = Tools_RecurringPaymentStripepay::createSetupIntent($privateKey);

        $this->_view->accountId = '';
        if(!empty($intent['intent'])) {
            $intentData = json_decode($intent['intent'], true);

            if(!empty($intentData)) {
                $intent = (object) $intentData;
                $this->_view->accountId = $accountId;
            }

        } elseif (!is_object($intent)) {
            return $this->_translator->translate('Stripe intent cannot be created!');
        }
        $this->_view->intent = $intent;
        $this->_view->publishKey = $publishKey;
        $this->_view->periods = $periods;
        $this->_view->organizationCountry = $shoppingConfig['country'];

        return $this->_view->render('single-product-payment.phtml');
    }

    protected function _processRecurringPeriods()
    {
        $index = array_search('period', $this->_options);
        $periods = [];
        if (!empty($index)) {
            for ($i = $index + 1; $i <= count($this->_options); $i++) {
                if (strstr(mb_strtolower($this->_options[$i]), self::MONTH_PERIOD)) {
                    $key = intval(substr($this->_options[$i], 0, strpos($this->_options[$i], self::MONTH_PERIOD)));
                    if ($key) {
                        $periods[] = $key . strtoupper(self::MONTH_PERIOD);
                    }
                } elseif (strpos(mb_strtolower($this->_options[$i]), self::YEAR_PERIOD)) {
                    $key = intval(substr($this->_options[$i], 0, strpos($this->_options[$i], self::YEAR_PERIOD)));
                    if ($key) {
                        $periods[] = $key . strtoupper(self::YEAR_PERIOD);
                    }
                }
            }
        }
        return array_intersect_key(self::SINGLE_PRODUCT_RECURRING_PERIODS, array_flip($periods));
    }

    public function checkIfQuoteSignedAction() {
        if ($this->_request->isPost()) {
            $cartStorage = Tools_ShoppingCart::getInstance();
            $cartId = $cartStorage->getCartId();
            $cartSession = Models_Mapper_CartSessionMapper::getInstance();
            $cart = $cartSession->find($cartId);

            if($cart != null){
                if ($cart->getGateway() === 'Quote') {
                    $quoteMapper = Quote_Models_Mapper_QuoteMapper::getInstance();
                    $quoteModel = $quoteMapper->findByCartId($cartId);
                    if ($quoteModel instanceof Quote_Models_Model_Quote) {
                        $signatureRequired = $quoteModel->getIsSignatureRequired();
                        if (!empty($signatureRequired) && empty($quoteModel->getIsQuoteSigned())) {
                            $this->_responseHelper->fail(array(
                                'error' => 1,
                                'generalError' => $this->_translator->translate('Please sign the quote!')
                            ));
                        } else {
                            $this->_responseHelper->success(array('error' => 0));
                        }
                    } else {
                        $this->_responseHelper->fail(array(
                            'error' => 1,
                            'generalError' => $this->_translator->translate('Quote not found!')
                        ));
                    }
                }
            }
        }
    }

    public function completeGoogleApplePaymentQuoteAction() {
        if ($this->_request->isPost()) {
            $requestParams = $this->_request->getParams();
            $stripeTransactionMapper = Stripepay_Models_Mappers_StripepayTransactionMapper::getInstance();
            $shoppingConfigMapper = Models_Mapper_ShoppingConfig::getInstance();
            $cartStorage = Tools_ShoppingCart::getInstance();
            $cartId = $cartStorage->getCartId();

            $cartSession = Models_Mapper_CartSessionMapper::getInstance();
            $cart = $cartSession->find($cartId);

            if($cart != null){
                if (!$cart instanceof Models_Model_CartSession) {
                    $cartStatus = $cart->getStatus();
                    $gateway = $cart->getGateway();
                    if (!in_array($cartStatus, $this->_allowedStatusesForNewOrder) && $gateway == 'Stripepay') {
                        $this->_responseHelper->fail(array(
                            'error' => 1,
                            'generalError' => $this->_translator->translate('We can\'t confirm your order at the moment due to a technical difficulty. If you do not receive an email in the coming hours confirming your purchase, please contact us')
                        ));
                    }
                }

                /*if ($cart->getGateway() === 'Quote') {
                    $quoteMapper = Quote_Models_Mapper_QuoteMapper::getInstance();
                    $quoteModel = $quoteMapper->findByCartId($cart->getId());
                    if ($quoteModel instanceof Quote_Models_Model_Quote) {
                        $signatureRequired = $quoteModel->getIsSignatureRequired();
                        if (!empty($signatureRequired) && empty($quoteModel->getIsQuoteSigned())) {
                            $this->_responseHelper->fail(array(
                                'error' => 1,
                                'generalError' => $this->_translator->translate('Please sign the quote!')
                            ));
                        }
                    }
                }*/

                $addressType = Models_Model_Customer::ADDRESS_TYPE_SHIPPING;
                $shippingAddressKey = Tools_ShoppingCart::getInstance()->getAddressKey($addressType);
                if ($shippingAddressKey != null) {
                    $customerShipping = Tools_ShoppingCart::getAddressById($shippingAddressKey);
                    $state = Tools_Geo::getStateById($customerShipping['state']);
                    $customerShipping['state'] = $state['state'];

                    $shippingAddressArray = array();
                    $shippingAddressArray['firstname'] = $customerShipping['firstname'];
                    $shippingAddressArray['lastname'] = $customerShipping['lastname'];
                    $shippingAddressArray['company'] = $customerShipping['company'];
                    $shippingAddressArray['email'] = $customerShipping['email'];
                    $shippingAddressArray['address1'] = $customerShipping['address1'];
                    $shippingAddressArray['address2'] = $customerShipping['address2'];
                    $shippingAddressArray['country'] = $customerShipping['country'];
                    $shippingAddressArray['city'] = $customerShipping['city'];
                    $shippingAddressArray['state'] = $customerShipping['state'];
                    $shippingAddressArray['zip'] = $customerShipping['zip'];
                    $shippingAddressArray['phone'] = isset($customerShipping['phone']) ? $customerShipping['phone'] : '';
                    $shippingAddressArray['phone_country_code_value'] = isset($customerShipping['phone_country_code_value']) ? $customerShipping['phone_country_code_value'] : '';
                    $shippingAddressArray['phonecountrycode'] = isset($customerShipping['phonecountrycode']) ? $customerShipping['phonecountrycode'] : '';

                    $addressId = $this->_addAddress($cart->getUserId(), $shippingAddressArray, $addressType);
                    $cart->setShippingAddressId($addressId);
                }

                $addressType = Models_Model_Customer::ADDRESS_TYPE_BILLING;
                $billingAddressKey = Tools_ShoppingCart::getInstance()->getAddressKey($addressType);
                if ($billingAddressKey != null) {
                    $customerBilling = Tools_ShoppingCart::getAddressById($billingAddressKey);
                    $state = Tools_Geo::getStateById($customerBilling['state']);
                    $customerBilling['state'] = $state['state'];

                    $billingAddressArray = array();
                    $billingAddressArray['firstname'] = $customerBilling['firstname'];
                    $billingAddressArray['lastname'] = $customerBilling['lastname'];
                    $billingAddressArray['company'] = $customerBilling['company'];
                    $billingAddressArray['email'] = $customerBilling['email'];
                    $billingAddressArray['address1'] = $customerBilling['address1'];
                    $billingAddressArray['address2'] = $customerBilling['address2'];
                    $billingAddressArray['country'] = $customerBilling['country'];
                    $billingAddressArray['city'] = $customerBilling['city'];
                    $billingAddressArray['state'] = $customerBilling['state'];
                    $billingAddressArray['zip'] = $customerBilling['zip'];
                    $billingAddressArray['phone'] = isset($customerBilling['phone']) ? $customerBilling['phone'] : '';
                    $billingAddressArray['phone_country_code_value'] = isset($customerBilling['phone_country_code_value']) ? $customerBilling['phone_country_code_value'] : '';
                    $billingAddressArray['phonecountrycode'] = isset($customerBilling['phonecountrycode']) ? $customerBilling['phonecountrycode'] : '';

                    $addressId = $this->_addAddress($cart->getUserId(), $billingAddressArray, $addressType);
                    $cart->setBillingAddressId($addressId);
                }

                $stripeTransactionModel = new Stripepay_Models_Models_StripepayTransactionModel();
                $stripeTransactionModel->setLiveMode($requestParams['liveMode']);
                $stripeTransactionModel->setCheckoutSessionId('');
                $stripeTransactionModel->setStripeCustomerId('');
                $stripeTransactionModel->setTotalTax($cart->getTotalTax());
                $stripeTransactionModel->setCartId($cartId);
                $stripeTransactionModel->setOrderShippingAddress($customerShipping ? json_encode($customerShipping) : '');
                $stripeTransactionModel->setOrderShippingAddress($customerBilling ? json_encode($customerBilling) : '');
                $stripeTransactionModel->setCurrency($shoppingConfigMapper->getConfigParam('currency'));
                $stripeTransactionModel->setTransactionType(Models_Model_CartSession::CART_STATUS_PROCESSING);
                $stripeTransactionModel->setFirstName($customerBilling['firstname'] ? $customerBilling['firstname'] : '');
                $stripeTransactionModel->setLastName($customerBilling['lastname'] ? $customerBilling['lastname'] : '');
                $stripeTransactionModel->setEmail($customerBilling['email'] ? $customerBilling['email'] : '');
                $stripeTransactionModel->setReason('');

                if($requestParams['amount'] != $requestParams['amountToPay']){
                    $reason =  $this->_translator->translate('Amount not right');
                    $stripeTransactionModel->setOrderAmount($requestParams['amount']);
                    $stripeTransactionModel->setStatus(Models_Model_CartSession::CART_STATUS_ERROR);
                    $stripeTransactionModel->setReason($reason);
                    $stripeTransactionMapper->save($stripeTransactionModel);
                    $this->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_ERROR, false, $reason);

                    $checkout = Tools_Misc::getCheckoutPage();
                    $this->_responseHelper->fail(array('error'=> 1,'redirect'=>$this->_websiteUrl.$checkout->getUrl()));
                } else {
                    //affiliate block
                    $enabledPlugins = Tools_Plugins_Tools::getEnabledPlugins();
                    foreach ($enabledPlugins as $plugin) {
                        if($plugin->getName() == 'seosambaaffiliatenetwork'){
                            $amountSale = $requestParams['amount']/100;
                            $idSale = $cartId;
                            $affiliate = Tools_Factory_PluginFactory::createPlugin('seosambaaffiliatenetwork');
                            $affiliateStatus = $affiliate->affiliateSale($idSale, $amountSale);
                        }
                    }
                    //affiliate end

                    if(empty($requestParams['hasError'])) {
                        $isPartial = false;
                        if (!empty($cart->getIsPartial())) {
                            $isPartial = true;
                        }

                        $partialPaymentType = $cart->getPartialType();

                        if ($isPartial === true) {
                            $partialPaidAmount = $cart->getPartialPaidAmount();
                            $cart->setIsPartial('1');
                            if (!empty($partialPaidAmount) && $partialPaidAmount !== '0.00') {
                                $cart->setPartialPaidAmount($cart->getTotal());
                                if ($partialPaymentType === Models_Model_CartSession::CART_PARTIAL_PAYMENT_TYPE_AMOUNT) {
                                    $amountToPayPartial = round($cart->getTotal() - $cart->getPartialPercentage(),
                                        2);
                                } else {
                                    $amountToPayPartial = round($cart->getTotal() - round(($cart->getTotal() * $cart->getPartialPercentage()) / 100,
                                            2), 2);
                                }
                                $updatePaymentStatus = Models_Model_CartSession::CART_STATUS_COMPLETED;
                            } else {
                                if ($partialPaymentType === Models_Model_CartSession::CART_PARTIAL_PAYMENT_TYPE_AMOUNT) {
                                    $amountToPayPartial = round($cart->getPartialPercentage(), 2);
                                } else {
                                    $amountToPayPartial = round(($cart->getTotal() * $cart->getPartialPercentage()) / 100, 2);
                                }
                                $cart->setPartialPaidAmount($amountToPayPartial);
                                $updatePaymentStatus = Models_Model_CartSession::CART_STATUS_PARTIAL;
                            }
                        }

                        $amountToPay = round($cart->getTotal(), 2);
                        if ($isPartial === true && !empty($amountToPayPartial)) {
                            $amountToPay = $amountToPayPartial;
                        }

                        $stripeTransactionModel->setAmount($cart->getTotal());

                        $paymentStatus = Models_Model_CartSession::CART_STATUS_COMPLETED;
                        if ($isPartial === true) {
                            $paymentStatus = $updatePaymentStatus;
                            $stripeTransactionModel->setAmount($amountToPayPartial);
                        }

                        $stripeTransactionModel->setStatus($paymentStatus);
                        $stripeTransactionModel->setPaymentMethod($requestParams['paymentMethod']);
                        $stripeTransactionModel->setPaymentIntent($requestParams['paymentIntent']);
                        $stripeTransactionModel->setIntentClientSecret($requestParams['clientSecret']);
                        $stripeTransactionModel->setOrderAmount($requestParams['amountToPay']);
                        $stripeTransactionModel->setCardType($requestParams['cardType']);
                        $stripeTransactionModel->setExpMonth($requestParams['expMonth']);
                        $stripeTransactionModel->setExpYear($requestParams['expYear']);
                        $stripeTransactionModel->setCardMask($requestParams['cardMask']);
                        $stripeTransactionModel->setPaymentType($requestParams['methodName']. ' ' . self::PAYMENT_TYPE_BUTTON . ' (Quote)');
                        $stripeTransactionModel->setEmailSent('1');
                        $stripeTransactionModel->setOrderDateFull(Tools_System_Tools::convertDateFromTimezone('now'));
                        $stripeTransactionModel->setTransactionType($requestParams['status']);
                        $stripeTransactionMapper->save($stripeTransactionModel);
                        Models_Mapper_CartSessionMapper::getInstance()->save($cart);
                        $this->updateCartStatus($cartId, $paymentStatus);

                        $cartSession = Models_Mapper_CartSessionMapper::getInstance()->find($cartId);
                        $this->_sessionHelper->storeCartSessionKey = $cartId;
                        $this->_sessionHelper->storeCartSessionConversionKey = $cartId;
                        if ($cartSession->getStatus() !== Models_Model_CartSession::CART_STATUS_PARTIAL) {
                            if ($isPartial === true) {
                                $cartSession->registerObserver(new Tools_Mail_Watchdog(array(
                                    'trigger' => Tools_StoreMailWatchdog::TRIGGER_STORE_PARTIALPAYMENT_SECOND
                                )));
                            } else {
                                $cartSession->registerObserver(new Tools_Mail_Watchdog(array(
                                    'trigger' => Tools_StoreMailWatchdog::TRIGGER_NEW_ORDER
                                )));
                            }
                        }

                        $shoppingConfig = $shoppingConfigMapper->getConfigParams();

                        if (!empty($shoppingConfig[Shopping::SHIPPING_IS_GIFT])) {
                            if (!empty($cartSession->getIsGift())) {
                                $cartSession->registerObserver(new Tools_Mail_Watchdog(array(
                                    'trigger' => Tools_StoreMailWatchdog::TRIGGER_STORE_GIFT_ORDER
                                )));
                            }
                        }

                        if ($cartSession->getStatus() === Models_Model_CartSession::CART_STATUS_PARTIAL) {
                            $cartSession->registerObserver(new Tools_Mail_Watchdog(array(
                                'trigger' => Tools_StoreMailWatchdog::TRIGGER_STORE_PARTIALPAYMENT
                            )));
                        }

                        if ($cartSession->getStatus() !== Models_Model_CartSession::CART_STATUS_PARTIAL) {
                            if (class_exists('Tools_AppsServiceWatchdog')) {
                                $cartSession->registerObserver(new Tools_AppsServiceWatchdog());
                            }
                        }

                        $cartSession->notifyObservers();

                        $currency = Zend_Registry::get('Zend_Currency');
                        if (empty($amountToPayPartial)) {
                            $message = $currency->toCurrency($amountToPay);
                        } else {
                            $message = $currency->toCurrency($amountToPayPartial);
                        }

                        $this->_responseHelper->success(array(
                            'error' => 0,
                            'generalSuccess' => $this->_translator->translate('Thank you for your payment of ') . $message
                        ));

                    } else {
                        $reason = $requestParams['message'];
                        $stripeTransactionModel->setOrderAmount($requestParams['amountToPay']);
                        $stripeTransactionModel->setStatus(Models_Model_CartSession::CART_STATUS_ERROR);
                        $stripeTransactionModel->setTransactionType($requestParams['status']);
                        $stripeTransactionModel->setReason($reason);
                        $stripeTransactionMapper->save($stripeTransactionModel);

                        if($requestParams['status'] != 'payment_not_found' && $requestParams['status'] != 'payment_failed') {
                            $this->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_ERROR, false, $reason);
                        }

                        $this->_responseHelper->fail(array(
                            'error' => 1,
                            'generalError' => $reason
                        ));
                    }
                }
            }
        }
    }

    /**
     * complete Google or Apple Payment
     *
     * @return void
     * @throws Exceptions_SeotoasterException
     * @throws Exceptions_SeotoasterPluginException
     * @throws Zend_Db_Table_Exception
     */
    public function completeGoogleApplePaymentAction()
    {
        $requestParams = $this->_request->getParams();
        $stripeTransactionMapper = Stripepay_Models_Mappers_StripepayTransactionMapper::getInstance();
        $transactionId = $requestParams['transactionId'];

        $stripeTransactionModel = $stripeTransactionMapper->find($transactionId);

        if($stripeTransactionModel instanceof Stripepay_Models_Models_StripepayTransactionModel) {
            $cartSession = Models_Mapper_CartSessionMapper::getInstance();
            $cartContent = $cartSession->find($stripeTransactionModel->getCartId());

            if($cartContent != null){
                $cartTotal = $cartContent->getTotal();
                if($requestParams['totalAmountOriginal']*100 != $cartTotal*100){
                    $reason =  $this->_translator->translate('Amount not right');
                    $stripeTransactionModel->setOrderAmount($requestParams['amount']);
                    $stripeTransactionModel->setStatus(Models_Model_CartSession::CART_STATUS_ERROR);
                    $stripeTransactionModel->setReason($reason);
                    $stripeTransactionMapper->save($stripeTransactionModel);
                    $this->updateCartStatus($stripeTransactionModel->getCartId(), Models_Model_CartSession::CART_STATUS_ERROR, false, $reason);

                    $checkout = Tools_Misc::getCheckoutPage();
                    $this->_responseHelper->fail(array('error'=> 1,'redirect'=>$this->_websiteUrl.$checkout->getUrl()));
                } else {
                    //affiliate block
                    $enabledPlugins = Tools_Plugins_Tools::getEnabledPlugins();
                    foreach ($enabledPlugins as $plugin) {
                        if($plugin->getName() == 'seosambaaffiliatenetwork'){
                            $amountSale = $requestParams['amount']/100;
                            $idSale = $stripeTransactionModel->getCartId();
                            $affiliate = Tools_Factory_PluginFactory::createPlugin('seosambaaffiliatenetwork');
                            $affiliateStatus = $affiliate->affiliateSale($idSale, $amountSale);
                        }
                    }
                    //affiliate end

                    if(empty($requestParams['hasError'])) {
                        $stripeTransactionModel->setStatus(Models_Model_CartSession::CART_STATUS_COMPLETED);
                        $stripeTransactionModel->setPaymentMethod($requestParams['paymentMethod']);
                        $stripeTransactionModel->setPaymentIntent($requestParams['paymentIntent']);
                        $stripeTransactionModel->setIntentClientSecret($requestParams['clientSecret']);
                        $stripeTransactionModel->setOrderAmount($requestParams['totalAmountOriginal']);
                        $stripeTransactionModel->setCardType($requestParams['cardType']);
                        $stripeTransactionModel->setExpMonth($requestParams['expMonth']);
                        $stripeTransactionModel->setExpYear($requestParams['expYear']);
                        $stripeTransactionModel->setCardMask($requestParams['cardMask']);
                        $stripeTransactionModel->setPaymentType($requestParams['methodName']. ' ' . self::PAYMENT_TYPE_BUTTON);
                        $stripeTransactionModel->setEmailSent('1');
                        $stripeTransactionModel->setOrderDateFull(Tools_System_Tools::convertDateFromTimezone('now'));
                        $stripeTransactionModel->setTransactionType($requestParams['status']);
                        $stripeTransactionMapper->save($stripeTransactionModel);
                        $this->updateCartStatus($stripeTransactionModel->getCartId(), Models_Model_CartSession::CART_STATUS_COMPLETED);
                        //$this->_redirector->gotoUrl($this->_websiteUrl.'plugin/shopping/run/thankyou/');

                        $payResult['error'] = 0;
                        $payResult['message'] = $this->_translator->translate('Payment successfully');
                        echo json_encode($payResult);

                    } else {
                        $reason = $requestParams['message'];
                        $stripeTransactionModel->setOrderAmount($requestParams['totalAmountOriginal']);
                        $stripeTransactionModel->setStatus(Models_Model_CartSession::CART_STATUS_ERROR);
                        $stripeTransactionModel->setTransactionType($requestParams['status']);
                        $stripeTransactionModel->setReason($reason);
                        $stripeTransactionMapper->save($stripeTransactionModel);

                        if($requestParams['status'] != 'payment_not_found' && $requestParams['status'] != 'payment_failed') {
                            $this->updateCartStatus($stripeTransactionModel->getCartId(), Models_Model_CartSession::CART_STATUS_ERROR, false, $reason);
                        }

                        $payResult['error'] = 1;
                        $payResult['message'] = $reason;
                        echo json_encode($payResult);

                    }
                }
            }
        }
    }

    public function makeStripeSessionAction()
    {
        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();

        $stripeConfig = $this->_getStripeConfig();
        $cartContent = Tools_ShoppingCart::getInstance();
        $cartId = $cartContent->getCartId();
        $cartSessionMapper = Models_Mapper_CartSessionMapper::getInstance();
        $cartSession = $cartSessionMapper->find($cartId);
        if (!$cartSession instanceof Models_Model_CartSession) {
            $cartStatus = $cartSession->getStatus();
            $gateway = $cartSession->getGateway();
            if (!in_array($cartStatus, $this->_allowedStatuses) && $gateway == 'Stripepay') {
                $payResult['error'] = 1;
                $payResult['message'] = $this->_translator->translate('We can\'t confirm your order at the moment due to a technical difficulty. If you do not receive an email in the coming hours confirming your purchase, please contact us');
                echo json_encode($payResult);
                exit;
            }
        }
        $this->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_PROCESSING);
        $currency = Models_Mapper_ShoppingConfig::getInstance()->getConfigParam('currency');

        $productList = array_values($cartContent->getContent());

        if (!empty($productList) && (!empty($stripeConfig['privateKey']) || !empty($stripepayOAuth))) {
            $lineItems = [];
            foreach ($productList as $product) {
                $lineItem['price_data'] = [];

                $lineItem['price_data']['product_data']['name'] = $product['name'];
                $lineItem['price_data']['product_data']['images'] = [$this->_websiteUrl . 'media/products/product/' . str_replace('products/', '', $product['photo'])];
                $lineItem['price_data']['product_data']['description'] = $product['shortDescription'];
                $lineItem['price_data']['unit_amount'] = round($product['taxPrice'], 2) * 100;
                $lineItem['price_data']['currency'] = strtolower($currency);
                $lineItem['quantity'] = $product['qty'];
                $lineItems[] = $lineItem;
            }
            $shippingAddressKey = Tools_ShoppingCart::getInstance()->getAddressKey(Models_Model_Customer::ADDRESS_TYPE_SHIPPING);
            if ($shippingAddressKey != null) {
                $customerShipping = Tools_ShoppingCart::getAddressById($shippingAddressKey);
                $state = Tools_Geo::getStateById($customerShipping['state']);
                $customerShipping['state'] = $state['state'];
            }

            $billingAddressArray = array();
            $billingAddressArray['firstname'] = $customerShipping['firstname'];
            $billingAddressArray['lastname'] = $customerShipping['lastname'];
            $billingAddressArray['company'] = '';
            $billingAddressArray['email'] = $customerShipping['email'];
            $billingAddressArray['address1'] = $customerShipping['address1'];
            $billingAddressArray['address2'] = '';
            $billingAddressArray['country'] = $customerShipping['country'];
            $billingAddressArray['city'] = $customerShipping['city'];
            $billingAddressArray['state'] = $customerShipping['state'];
            $billingAddressArray['zip'] = $customerShipping['zip'];
            $billingAddressArray['phone'] = isset($customerShipping['phone']) ? $customerShipping['phone'] : '';
            $billingAddressArray['phone_country_code_value'] = isset($customerShipping['phone_country_code_value']) ? $customerShipping['phone_country_code_value'] : '';
            $billingAddressArray['phonecountrycode'] = isset($customerShipping['phonecountrycode']) ? $customerShipping['phonecountrycode'] : '';

            $addressId = $this->_addAddress($cartSession->getUserId(), $billingAddressArray, Models_Model_Customer::ADDRESS_TYPE_BILLING);
            $cartSession->setBillingAddressId($addressId);
            $cartSessionMapper->save($cartSession);

            $cartContent = Tools_ShoppingCart::getInstance();
            $cartParams = $cartContent->calculate();

            $shippingOptions = array();
            $shippingData = $cartContent->getShippingData();
            $shippingPrice = round($cartParams['shipping'], 2);
            $shippingTax = round($cartParams['shippingTax'], 2);

            if(!empty($shippingData) && !empty($shippingPrice)) {
                $shippingAmount = $shippingPrice;

                if(!empty($shippingTax)) {
                    $shippingAmount += $shippingTax;
                }

                $shippingOptions[]['shipping_rate_data'] = array(
                    'type' => 'fixed_amount',
                    'fixed_amount' => array(
                        'amount' => $shippingAmount * 100,
                        'currency' => strtolower($currency),
                    ),
                    'display_name' => $shippingData['service'].'-'.$shippingData['type'],
                );
            }

            $customerEmailShipping = isset($customerShipping['email']) ? $customerShipping['email'] : '';

            if(!empty($stripepayOAuth)){
                //return Obj
                $configData = Application_Model_Mappers_ConfigMapper::getInstance()->getConfig();
                $websiteId = $configData['websiteId'];
                $metadata = array();
                $metadata['websiteId'] = $websiteId;
                $metadata['cartId'] = $cartId;

                $responceStripeSession = Tools_StripeOAuthTools::createStripeSessionMojo(
                    array('card'),
                    $shippingOptions,
                    $lineItems,
                    $customerEmailShipping,
                    $this->_websiteUrl . 'plugin/shopping/run/thankyou?session_id={CHECKOUT_SESSION_ID}',
                    $this->_websiteUrl . 'check-out.html',
                    $metadata
                );
                if(!empty($responceStripeSession)) {
                    $responceStripeSessionData = json_decode($responceStripeSession, true);

                    if(!empty($responceStripeSessionData)) {
                        $stripeSession = (object) $responceStripeSessionData;
                    }
                }

                if (!is_object($stripeSession)) {
                    return $this->_translator->translate('Stripe session cannot be created!');
                }
            } else {
                Stripe::setApiKey($stripeConfig['privateKey']);
                $stripeSession = Session::create([
                    'payment_method_types' => ['card'],
                    'shipping_options' => $shippingOptions,
                    'mode' => 'payment',
                    'line_items' => $lineItems,
                    'customer_email' => $customerEmailShipping,
                    'success_url' => $this->_websiteUrl . 'plugin/shopping/run/thankyou?session_id={CHECKOUT_SESSION_ID}',
                    'cancel_url' => $this->_websiteUrl . 'check-out.html',
                ]);
            }
            if (isset($stripeSession->id)) {
                try {
                    $stripeTransactionMapper = Stripepay_Models_Mappers_StripepayTransactionMapper::getInstance();
                    $stripeTransactionModel = new Stripepay_Models_Models_StripepayTransactionModel();
                    $stripeTransactionModel->setCheckoutSessionId($stripeSession->id);
                    $stripeTransactionModel->setStripeCustomerId('');
                    $stripeTransactionModel->setPaymentIntent('');
                    $stripeTransactionModel->setStatus(self::TRANSACTION_STATUS_PROCESSING);
                    $stripeTransactionModel->setAmount($cartContent->getTotal());
                    $stripeTransactionModel->setTotalTax($cartContent->getTotalTax());
                    $stripeTransactionModel->setCartId($cartContent->getCartId());
                    $stripeTransactionModel->setOrderShippingAddress($customerShipping ? json_encode($customerShipping) : '');
                    $stripeTransactionModel->setOrderDateFull(Tools_System_Tools::convertDateFromTimezone('now'));
                    $stripeTransactionModel->setCurrency(strtolower($currency));
                    $stripeTransactionModel->setTransactionType(self::TRANSACTION_TYPE_SESSION_CHECKOUT);
                    $stripeTransactionModel->setPaymentType(self::PAYMENT_TYPE_BUTTON);
                    $stripeTransactionModel->setFirstName($customerShipping['firstname']);
                    $stripeTransactionModel->setLastName($customerShipping['lastname']);
                    $stripeTransactionModel->setEmail($customerShipping['email']);

                    $stripeTransactionMapper->save($stripeTransactionModel);
                } catch (Exception $e) {
                    error_log($e->getMessage());
                }
                echo json_encode(array('error' => 0, 'sessionId' => $stripeSession->id));
                return;
            } else {
                echo json_encode(array('error' => 1, 'sessionId' => $stripeSession->id));
                return;
            }

        }

    }

    /**
     * Prepare states list
     *
     * @param string $country country name
     * @return array
     */
    public function prepareStateList($country)
    {
        $pairs = true;
        $stateData = Tools_Geo::getState($country, $pairs);
        if (!empty ($stateData)) {
            $states = '';
            foreach ($stateData as $short => $state) {
                $states .= '<option value="' . $short . '">' . $state . '</option>';
            }

            return array('error' => '0', $states);
        }

        return array('error' => '1');

    }

    /**
     * Prepare country list
     *
     * @return array
     */
    private function _prepareCountryList()
    {
        $data = Tools_Geo::getCountries(true);
        asort($data);

        return $data;
    }

    public function payCreditCardAction()
    {
        if ($this->_request->isPost()) {

            $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();

            $privateKey = '';
            if(empty($stripepayOAuth)) {
                $stripeConfig = $this->_getStripeConfig();
                $privateKey = $stripeConfig['privateKey'];
            }

            $data = filter_var_array($this->_request->getParams(), FILTER_SANITIZE_STRING);
            $data = array_filter($data);
            if (empty($data)) {
                $this->_responseHelper->fail('');
            }
            $cartStorage = Tools_ShoppingCart::getInstance();
            $cartId = $cartStorage->getCartId();
            $cartSessionMapper = Models_Mapper_CartSessionMapper::getInstance();
            $cartSession = $cartSessionMapper->find($cartId);
            $payResult = [];
            if (!$cartSession instanceof Models_Model_CartSession) {
                $cartStatus = $cartSession->getStatus();
                $gateway = $cartSession->getGateway();
                if (!in_array($cartStatus, $this->_allowedStatuses) && $gateway == 'Stripepay') {
                    $payResult['error'] = 1;
                    $payResult['message'] = $this->_translator->translate('We can\'t confirm your order at the moment due to a technical difficulty. If you do not receive an email in the coming hours confirming your purchase, please contact us');
                    echo json_encode($payResult);
                    exit;
                }
            }
            $this->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_PROCESSING);

            $cartSummary = $cartStorage->calculate();
            $cartSummary['totalTax'] = round($cartSummary['totalTax'], 2);
            $cartSummary['shipping'] = round($cartSummary['shipping'], 2);
            $cartSummary['total'] = round($cartSummary['total'], 2);
            $amount = round($this->_request->getParam('credit-card-amt'), 2);
            $checkout = Tools_Misc::getCheckoutPage();
            if ($cartSummary['total'] !== $amount) {
                $this->_responseHelper->fail(array(
                    'error' => 1,
                    'redirect' => $this->_websiteUrl . $checkout->getUrl()
                ));
            }

            $state = Tools_Geo::getStateById($data['stripe-credit-card-state']);
            if (!empty($state)) {
                $data['state'] = $state['state'];
            }

            $countryPhoneCodeValue = '+' . Zend_Locale::getTranslation($data['phonecountrycodes'], 'phoneToTerritory');
            $phone = preg_replace("/[^0-9]/", '', $data['stripe-phone']);
            $shippingAddressKey = $cartStorage->getShippingAddressKey();
            if ($shippingAddressKey != null) {
                $customerShipping = Tools_ShoppingCart::getAddressById($shippingAddressKey);
                $state = Tools_Geo::getStateById($customerShipping['state']);
                $customerShipping['state'] = $state['state'];
            }
            $customer = Tools_ShoppingCart::getInstance()->getCustomer();
            $userEmail = $customer->getEmail();
            $userFullName = $customer->getFullName();
            $userFullInfo = explode(' ', $userFullName);
            $userFirstName = $userFullInfo[0];
            $userLastName = '';
            if (!empty($userFullInfo[1])) {
                $userLastName = $userFullInfo[1];
            }
            //saving billing address//
            $addressType = Models_Model_Customer::ADDRESS_TYPE_BILLING;
            $billCountryResult = '';
            $billStateResult = '';
            $countries = Tools_Geo::getCountries(true);
            $pCountryCode = $data['stripe-credit-card-country'];
            $pAddressState = $data['stripe-credit-card-state'];
            if (isset($pCountryCode)) {
                if (in_array($pCountryCode, $countries)) {
                    $billCountryResult = array_search($pCountryCode, $countries);
                }
                if (array_key_exists($pCountryCode, $countries)) {
                    $billCountryResult = $pCountryCode;
                }
            }
            $states = Tools_Geo::getState(null, true);
            if (isset($pAddressState)) {
                if (in_array($pAddressState, $states)) {
                    $billStateResult = array_search($pAddressState, $states);
                } elseif (!empty($pAddressState)) {
                    $billStateResult = $pAddressState;
                }
            }
            $billingAddressArray = array();
            $billingAddressArray['firstname'] = $data['stripe-first_name'];
            $billingAddressArray['lastname'] = $data['stripe-last_name'];
            $billingAddressArray['company'] = '';
            $billingAddressArray['email'] = $data['stripe-email'];
            $billingAddressArray['address1'] = $data['stripe-address'];
            $billingAddressArray['address2'] = '';
            $billingAddressArray['country'] = $billCountryResult;
            $billingAddressArray['city'] = isset($data['stripe-city']) ? $data['stripe-city'] : '';
            $billingAddressArray['state'] = $billStateResult;
            $billingAddressArray['zip'] = isset($data['stripe-zip']) ? $data['stripe-zip'] : '';
            $billingAddressArray['phone'] = isset($phone) ? $phone : '';
            $billingAddressArray['phone_country_code_value'] = isset($countryPhoneCodeValue) ? $countryPhoneCodeValue : '';
            $billingAddressArray['phonecountrycode'] = isset($data['phonecountrycodes']) ? $data['phonecountrycodes'] : '';

            $addressId = $this->_addAddress($cartSession->getUserId(), $billingAddressArray, $addressType);
            $cartSession->setBillingAddressId($addressId);
            $cartSessionMapper->save($cartSession);
            //end of saving billing address//

            $paymentData = array();
            $paymentData['totalTax'] = $cartSummary['totalTax'];
            $paymentData['cartId'] = $cartId;
            $paymentData['customerShipping'] = $customerShipping ? json_encode($customerShipping) : '';
            $paymentData['billingAddress'] = json_encode($billingAddressArray);
            $paymentData['userFirstName'] = $userFirstName;
            $paymentData['userLastName'] = $userLastName;
            $paymentData['userEmail'] = $userEmail;
            $paymentData['amount'] = $data['credit-card-amt'];
            $paymentData['stripeToken'] = $data['stripeToken'];

            $recurrent = $cartStorage->getRecurringPaymentType();
            //affiliate block
            $enabledPlugins = Tools_Plugins_Tools::getEnabledPlugins();
            foreach ($enabledPlugins as $plugin) {
                if ($plugin->getName() == 'seosambaaffiliatenetwork') {
                    $affiliate = Tools_Factory_PluginFactory::createPlugin('seosambaaffiliatenetwork');
                    $affiliateStatus = $affiliate->affiliateSave($cartId, 'stripepay', 'creditcard');
                }
            }
            //affiliate end
            if (!empty($recurrent)) {
                if (!empty($paymentMethod = $data['paymentMethod'])) {
                    try {
                        $gw = new Tools_StripeRecurrent($privateKey, $paymentMethod, $paymentData);
                        $payResult = $gw->processPaymentIntent(self::PAYMENT_TYPE_CART);
                    } catch (Exception $e) {
                        $payResult['error'] = 1;
                        $payResult['message'] = $this->_translator->translate(self::ERROR_PAYMENT_COMMON);
                    }
                } else {
                    $payResult['error'] = 1;
                    $payResult['message'] = $this->_translator->translate(self::ERROR_PAYMENT_METHOD_NOT_FOUND);
                }

            } else {
                if(!empty($data['intentId'])) {
                    if(!empty($stripepayOAuth)) {
                        //return Obj
                        $responcePaymentIntent = Tools_StripeOAuthTools::retrievePaymentIntentMojo($data['intentId']);
                        if(!empty($responcePaymentIntent)) {
                            $responcePaymentIntentData = json_decode($responcePaymentIntent, true);

                            if(!empty($responcePaymentIntentData)) {
                                $intent = (object) $responcePaymentIntentData;
                            }
                        }

                        if (!is_object($intent)) {
                            return $this->_translator->translate('Stripe intent cannot be retrieved!');
                        }

                    } else {
                        Stripe::setApiKey($privateKey);
                        $intent = PaymentIntent::retrieve($data['intentId']);
                    }

                    if(!empty($intent)) {
                        $paymentAmount = $intent->amount;

                        if(!empty($stripepayOAuth)) {
                            $intentCartId = $intent->metadata['cartId'];
                        } else {
                            $intentCartId = $intent->metadata->cartId;
                        }

                        $paymentStatus = $intent->status;//succeeded

                        $payResult['error'] = 0;
                        if($paymentStatus == 'succeeded') {
                            if($cartId != $intentCartId) {
                                $payResult['error'] = 1;
                                $payResult['message'] = $this->_translator->translate('cartId ') . ' ' . $intentCartId . ' ' . $this->_translator->translate('in intent data does not match in cart') . ' ' . $cartId;
                            }

                            if(intval(strval($cartSummary['total'] * 100)) != $paymentAmount) {
                                $payResult['error'] = 1;
                                $payResult['message'] = $this->_translator->translate('Cart total not matched');
                            }
                        } else {
                            $payResult['error'] = 1;
                            $payResult['message'] = $this->_translator->translate('Payment not succeeded.');
                        }

                        if($payResult['error'] == 0) {
                            Tools_StripeRegular::saveIntentTransactionLog($intent, $paymentData);
                        }
                    }
                } else {
                    $payResult['error'] = 1;
                    $payResult['message'] = $this->_translator->translate('Payment intent not found');
                }
            }

            if (empty($payResult)) {
                $payResult['error'] = 1;
                $payResult['message'] = $this->_translator->translate(self::ERROR_PAYMENT_CONNECTION_ERROR);
                $this->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_ERROR, false, $payResult['message']);
                echo json_encode($payResult);
                return;
            } elseif ($payResult['error'] == 0) {
                $this->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_COMPLETED);
                if (!empty($recurrent) && isset($payResult['paymentMethodId'])) {
                    $subscriptionId = $cartId . '_@@_' . $payResult['paymentMethodId'];
                    Tools_RecurringPaymentTools::createRecurringPaymentInfo($cartId, 65535, $subscriptionId, '',
                        $recurrent, $amount, 'stripepay', $amount, '', self::ACTIVE_STATUS, 1, 1);
                }
                $this->_sessionHelper->storeCartSessionKey = $cartId;
                $this->_sessionHelper->storeCartSessionConversionKey = $cartId;
            } else {
                $this->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_ERROR, false, isset($payResult['message']) ? $payResult['message'] : $this->_translator->translate(self::ERROR_PAYMENT_COMMON));
            }
            echo json_encode($payResult);
            return;
        }
    }

    public function payCreditCardQuoteRecurringAction() {
        if ($this->_request->isPost()) {
            $tokenToValidate = $this->_request->getParam(Tools_System_Tools::CSRF_SECURE_TOKEN, false);
            $valid = Tools_System_Tools::validateToken($tokenToValidate, Shopping::SHOPPING_SECURE_TOKEN);
            if (!$valid) {
                $this->_responseHelper->fail('');
            }

            $stripeConfig = $this->_getStripeConfig();
            $privateKey = $stripeConfig['privateKey'];
            $payResult = [];
            $data = filter_var_array($this->_request->getParams(), FILTER_SANITIZE_STRING);
            $data = array_filter($data);
            if (empty($data)) {
                $this->_responseHelper->fail('');
            }

            $cartStorage = Tools_ShoppingCart::getInstance();
            $cartId = $cartStorage->getCartId();
            $cart = Models_Mapper_CartSessionMapper::getInstance()->find($cartId);

            if (!$cart instanceof Models_Model_CartSession) {
                $cartStatus = $cart->getStatus();
                $gateway = $cart->getGateway();
                if (!in_array($cartStatus, $this->_allowedStatusesCheckout) && $gateway == 'Stripepay') {
                    $this->_responseHelper->fail(array(
                        'error' => 1,
                        'generalError' => $this->_translator->translate('We can\'t confirm your order at the moment due to a technical difficulty. If you do not receive an email in the coming hours confirming your purchase, please contact us')
                    ));
                }
            }

            if ($cart->getGateway() === 'Quote') {
                $quoteMapper = Quote_Models_Mapper_QuoteMapper::getInstance();
                $quoteModel = $quoteMapper->findByCartId($cart->getId());
                if ($quoteModel instanceof Quote_Models_Model_Quote) {
                    $signatureRequired = $quoteModel->getIsSignatureRequired();
                    if (!empty($signatureRequired) && empty($quoteModel->getIsQuoteSigned())) {
                        $this->_responseHelper->fail(array(
                            'error' => 1,
                            'generalError' => $this->_translator->translate('Please sign the quote!')
                        ));
                    }

                    $paymentType = $quoteModel->getPaymentType();
                    if ($paymentType === Quote_Models_Model_Quote::PAYMENT_TYPE_PARTIAL_PAYMENT) {
                        $this->_responseHelper->fail(array(
                            'error' => 1,
                            'generalError' => $this->_translator->translate('This payment is used for recurring payments, not for partial!')
                        ));
                    }
                }
            }

            $cartSummary = $cartStorage->calculate();
            $cartSummary['totalTax'] = round($cart->getTotalTax(), 2);
            $cartSummary['shipping'] = round($cart->getShippingPrice(), 2);
            $cartSummary['total'] = round($cart->getTotal(), 2);
            $amount = round($this->_request->getParam('credit-card-amt'), 2);
            $checkout = Tools_Misc::getCheckoutPage();
            if ($cartSummary['total'] !== $amount) {
                $this->_responseHelper->fail(array(
                    'error' => 1,
                    'redirect' => $this->_websiteUrl . $checkout->getUrl()
                ));
            }

            $customer = Tools_ShoppingCart::getInstance()->getCustomer();
            $addressType = Models_Model_Customer::ADDRESS_TYPE_SHIPPING;
            $shippingAddressKey = Tools_ShoppingCart::getInstance()->getAddressKey($addressType);
            if ($shippingAddressKey != null) {
                $customerShipping = Tools_ShoppingCart::getAddressById($shippingAddressKey);

                $shippingAddressArray = array();
                $shippingAddressArray['firstname'] = $customerShipping['firstname'];
                $shippingAddressArray['lastname'] = $customerShipping['lastname'];
                $shippingAddressArray['company'] = $customerShipping['company'];
                $shippingAddressArray['email'] = $customerShipping['email'];
                $shippingAddressArray['address1'] = $customerShipping['address1'];
                $shippingAddressArray['address2'] = $customerShipping['address2'];
                $shippingAddressArray['country'] = $customerShipping['country'];
                $shippingAddressArray['city'] = $customerShipping['city'];
                $shippingAddressArray['state'] = $customerShipping['state'];
                $shippingAddressArray['zip'] = $customerShipping['zip'];
                $shippingAddressArray['phone'] = isset($customerShipping['phone']) ? $customerShipping['phone'] : '';
                $shippingAddressArray['phone_country_code_value'] = isset($customerShipping['phone_country_code_value']) ? $customerShipping['phone_country_code_value'] : '';
                $shippingAddressArray['phonecountrycode'] = isset($customerShipping['phonecountrycode']) ? $customerShipping['phonecountrycode'] : '';

                $addressIdShipping = $this->_addAddress($cart->getUserId(), $shippingAddressArray, $addressType);
                $cart->setShippingAddressId($addressIdShipping);
            }

            $countryPhoneCodeValue = '+'.Zend_Locale::getTranslation($data['phonecountrycode'], 'phoneToTerritory');
            $phone = preg_replace("/[^0-9]/", '', $data['stripe-phone']);

            $addressType = Models_Model_Customer::ADDRESS_TYPE_BILLING;
            $billingAddressKey = Tools_ShoppingCart::getInstance()->getAddressKey($addressType);
            if ($billingAddressKey != null) {
                $customerBilling = Tools_ShoppingCart::getAddressById($billingAddressKey);

                $billingAddressArray = array();
                $billingAddressArray['firstname'] = isset($data['stripe-first_name']) ? $data['stripe-first_name'] : '';
                $billingAddressArray['lastname'] = isset($data['stripe-last_name']) ? $data['stripe-last_name'] : '';
                $billingAddressArray['company'] = $customerBilling['company'];
                $billingAddressArray['email'] = isset($data['stripe-email']) ? $data['stripe-email'] : '';
                $billingAddressArray['address1'] = isset($data['stripe-address']) ? $data['stripe-address'] : '';
                $billingAddressArray['address2'] = isset($data['stripe-address2']) ? $data['stripe-address2'] : '';
                $billingAddressArray['country'] = isset($data['stripe-credit-card-country']) ? $data['stripe-credit-card-country'] : '';
                $billingAddressArray['city'] = isset($data['stripe-city']) ? $data['stripe-city'] : '';
                $billingAddressArray['state'] = isset($data['stripe-credit-card-state']) ? $data['stripe-credit-card-state'] : '';
                $billingAddressArray['zip'] = isset($data['stripe-zip']) ? $data['stripe-zip'] : $customerBilling['zip'];
                $billingAddressArray['phone'] = isset($phone) ? $phone : $customerBilling['phone'];
                $billingAddressArray['phone_country_code_value'] = isset($countryPhoneCodeValue) ? $countryPhoneCodeValue : $customerBilling['phone_country_code_value'];
                $billingAddressArray['phonecountrycode'] = isset($data['phonecountrycodes']) ? $data['phonecountrycodes'] : $customerBilling['phonecountrycode'];

                $addressIdBilling = $this->_addAddress($cart->getUserId(), $billingAddressArray, $addressType);
                $cart->setBillingAddressId($addressIdBilling);
            }

            $paymentData = [];
            $paymentData['totalTax'] = $cart->getTotalTax();
            $paymentData['cartId'] = $cartId;
            $paymentData['userFirstName'] = isset($data['stripe-first_name']) ? $data['stripe-first_name'] : '';
            $paymentData['userLastName'] = isset($data['stripe-last_name']) ? $data['stripe-last_name'] : '';
            $paymentData['userEmail'] = isset($data['stripe-email']) ? $data['stripe-email'] : '';
            $paymentData['amount'] = $cartSummary['total'];
            $paymentData['stripeToken'] = $data['stripeToken'];
            $paymentData['customerShipping'] = !empty($addressIdShipping) ? json_encode(Tools_ShoppingCart::getAddressById($addressIdShipping)) : '';
            $paymentData['billingAddress'] = !empty($addressIdBilling) ? json_encode(Tools_ShoppingCart::getAddressById($addressIdBilling)) : '';

            $recurrent = $data['recurring-payment-period-type'];
            if (!empty($recurrent) && $recurrent !== '0') {
                if (strstr(mb_strtolower($recurrent), self::MONTH_PERIOD)) {
                    $qty = intval(substr($recurrent, 0, strpos(strtolower($recurrent), self::MONTH_PERIOD)));
                    $recurrent = '+' . $qty . ' ' . self::MONTH_PERIOD;
                } elseif (strstr(mb_strtolower($recurrent), self::YEAR_PERIOD)) {
                    $qty = intval(substr($recurrent, 0, strpos(strtolower($recurrent), self::YEAR_PERIOD)));
                    $recurrent = '+' . $qty . ' ' . self::YEAR_PERIOD;
                }
            } else {
                $recurrent = null;
            }

            //affiliate block
            $enabledPlugins = Tools_Plugins_Tools::getEnabledPlugins();
            foreach ($enabledPlugins as $plugin) {
                if ($plugin->getName() == 'seosambaaffiliatenetwork') {
                    $affiliate = Tools_Factory_PluginFactory::createPlugin('seosambaaffiliatenetwork');
                    $affiliateStatus = $affiliate->affiliateSave($cartId, 'stripepay', 'creditcard');
                }
            }
            //affiliate end

            Models_Mapper_CartSessionMapper::getInstance()->save($cart);

            $paymentMethod = $data['paymentMethod'];
            if (!empty($paymentMethod)) {
                try {
                    $gw = new Tools_StripeRecurrent($privateKey, $paymentMethod, $paymentData);
                    $payResult = $gw->processPaymentIntent(self::PAYMENT_TYPE_CART_QUOTE);
                } catch (Exception $e) {
                    $payResult['error'] = 1;
                    $payResult['message'] = $this->_translator->translate(self::ERROR_PAYMENT_COMMON);
                }
            } else {
                $payResult['error'] = 1;
                $payResult['message'] = $this->_translator->translate(self::ERROR_PAYMENT_METHOD_NOT_FOUND);
            }

            if (empty($payResult)) {
                $this->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_ERROR, false, $this->_translator->translate(self::ERROR_PAYMENT_CONNECTION_ERROR));
                $this->_responseHelper->fail(array(
                    'error' => 1,
                    'generalError' => $this->_translator->translate(self::ERROR_PAYMENT_CONNECTION_ERROR)
                ));
            } elseif ($payResult['error'] == 0) {
                $this->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_COMPLETED);
                if (!empty($recurrent) && isset($payResult['paymentMethodId'])) {
                    $subscriptionId = $cartId . '_@@_' . $payResult['paymentMethodId'];
                    Tools_RecurringPaymentTools::createRecurringPaymentInfo($cartId, 65535, $subscriptionId, '',
                        $recurrent, $amount, 'stripepay', $amount, '', self::ACTIVE_STATUS, 1, 1);
                }
                $this->_sessionHelper->storeCartSessionKey = $cartId;
                $this->_sessionHelper->storeCartSessionConversionKey = $cartId;

                $shoppingConfig = Models_Mapper_ShoppingConfig::getInstance()->getConfigParams();
                $cartSession = Models_Mapper_CartSessionMapper::getInstance()->find($cartId);

                if (!empty($shoppingConfig[Shopping::SHIPPING_IS_GIFT])) {
                    if (!empty($cartSession->getIsGift())) {
                        $cartSession->registerObserver(new Tools_Mail_Watchdog(array(
                            'trigger' => Tools_StoreMailWatchdog::TRIGGER_STORE_GIFT_ORDER
                        )));
                    }
                }

                if ($cartSession->getStatus() !== Models_Model_CartSession::CART_STATUS_PARTIAL) {
                    $cartSession->registerObserver(new Tools_Mail_Watchdog(array(
                        'trigger' => Tools_StoreMailWatchdog::TRIGGER_NEW_ORDER
                    )));

                    if (class_exists('Tools_AppsServiceWatchdog')) {
                        $cartSession->registerObserver(new Tools_AppsServiceWatchdog());
                    }
                }

                $cartSession->notifyObservers();

                $this->_responseHelper->success(array(
                    'error' => 0,
                    'generalSuccess' => $this->_translator->translate('Successful pay')
                ));
            } else {
                $this->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_ERROR, false, $this->_translator->translate($payResult['message']));
                $this->_responseHelper->fail(array(
                    'error' => 1,
                    'generalError' => $this->_translator->translate($payResult['message'])
                ));
            }
        }
    }

    public function paySingleProductAction()
    {
        if ($this->_request->isPost()) {
            $tokenToValidate = $this->_request->getParam(Tools_System_Tools::CSRF_SECURE_TOKEN, false);
            $valid = Tools_System_Tools::validateToken($tokenToValidate, Shopping::SHOPPING_SECURE_TOKEN);
            if (!$valid) {
                $this->_responseHelper->fail('');
            }
            $stripeConfig = $this->_getStripeConfig();
            $privateKey = $stripeConfig['privateKey'];
            $payResult = [];
            $data = filter_var_array($this->_request->getParams(), FILTER_SANITIZE_STRING);
            $data = array_filter($data);
            if (empty($data)) {
                $this->_responseHelper->fail('');
            }
            $productQty = !empty($data['product-qty']) ? $data['product-qty'] : 1;
            $cartSession = Tools_ShoppingCart::getInstance();
            $productMapper = Models_Mapper_ProductMapper::getInstance();
            $product = $productMapper->find($data['product-id']);
            if (!$product instanceof Models_Model_Product) {
                $payResult['error'] = 1;
                $payResult['message'] = $this->_translator->translate(self::ERROR_PRODUCT_NOT_FOUND);
                echo json_encode($payResult);
                return;
            }
            $cartSession->clean();
            $cartSession->setContent(array());
            $cartSession->setCartId(null);

            $options = [];
            $productOptions = $data['productOptions'];
            if(!empty($productOptions)) {
                $options = Tools_StripeRegular::parseOptionsString($productOptions);
            }

            $cartSession->add($product, $options, $productQty);
            $cartSession->calculate(true);
            $cartSession->save();

            $session = Zend_Controller_Action_HelperBroker::getStaticHelper('session');
            $currentLoggedUser = $session->getCurrentUser();

            $billingAddress = array();
            $addressType = Models_Model_Customer::ADDRESS_TYPE_BILLING;
            $billingAddress['firstname'] = isset($data['stripe-first_name']) ? $data['stripe-first_name'] : '';
            $billingAddress['lastname'] = isset($data['stripe-last_name']) ? $data['stripe-last_name'] : '';
            $billingAddress['company'] = '';
            $billingAddress['email'] = isset($data['stripe-email']) ? $data['stripe-email'] : '';
            $billingAddress['address1'] = isset($data['stripe-address']) ? $data['stripe-address'] : '';
            $billingAddress['address2'] = '';
            $billingAddress['country'] = isset($data['stripe-credit-card-country']) ? $data['stripe-credit-card-country'] : '';
            $billingAddress['city'] = isset($data['stripe-city']) ? $data['stripe-city'] : '';
            $billingAddress['state'] = isset($data['stripe-credit-card-state']) ? $data['stripe-credit-card-state'] : '';
            $billingAddress['zip'] = isset($data['stripe-zip']) ? $data['stripe-zip'] : '';
            $billingAddress['phone'] = isset($data['stripe-phone']) ? $data['stripe-phone'] : '';
            $billingAddress['phone_country_code_value'] = '';
            $billingAddress['phonecountrycode'] = isset($data['phonecountrycodes']) ? $data['phonecountrycodes'] : '';
            $customerData = [];
            if ($currentLoggedUser instanceof Application_Model_Models_User && $currentLoggedUser->getId()) {
                $customerData['email'] = $currentLoggedUser->getEmail();
            } else {
                $customerData['firstname'] = $data['stripe-first_name'];
                $customerData['lastname'] = $data['stripe-last_name'];
                $customerData['email'] = $data['stripe-email'];
                $customerData['mobilecountrycode'] = $data['phonecountrycodes'];
                $customerData['mobile'] = $data['stripe-phone'];
            }
            $isNewCustomer = null === Models_Mapper_CustomerMapper::getInstance()->findByEmail($data['stripe-email']);
            $customer = Shopping::processCustomer($customerData);
            if ($customer->getId()) {
                $cartSession->setCustomerId($customer->getId())->calculate(true);
                $addressId = Models_Mapper_CustomerMapper::getInstance()->addAddress($customer, $billingAddress,
                    $addressType);
                $cartSession->setBillingAddressKey($addressId);
                $cartSession->setShippingAddressKey($addressId);
                $cartSession->save()->saveCartSession($customer);
                /*if ($isNewCustomer) {
                    $customer->registerObserver(new Tools_Mail_Watchdog(array(
                        'trigger' => Tools_StoreMailWatchdog::TRIGGER_NEW_CUSTOMER
                    )));
                    $customer->notifyObservers();
                }*/
            }

            $paymentData = [];
            $cartId = $cartSession->getCartId();
            $amount = $cartSession->getTotal();
            $amount = round($amount, 2);
            $customerFullName = explode(' ', $customer->getFullName());

            $paymentData['totalTax'] = $cartSession->getTotalTax();
            $paymentData['cartId'] = $cartId;
            $paymentData['customerShipping'] = '';
            $paymentData['billingAddress'] = '';
            $paymentData['userFirstName'] = is_array($customerFullName) ? $customerFullName[0] : '';
            $paymentData['userLastName'] = isset($customerFullName[1]) ? $customerFullName[1] : '';
            $paymentData['userEmail'] = $customer->getEmail();
            $paymentData['amount'] = $amount;
            $paymentData['stripeToken'] = $data['stripeToken'];
            $paymentData['customerShipping'] = null !== $addressId ? json_encode(Tools_ShoppingCart::getAddressById($addressId)) : '';
            $paymentData['billingAddress'] = $paymentData['customerShipping'];

            $recurrent = $data['recurring-payment-period-type'];
            if (!empty($recurrent) && $recurrent !== '0') {
                if (strstr(mb_strtolower($recurrent), self::MONTH_PERIOD)) {
                    $qty = intval(substr($recurrent, 0, strpos(strtolower($recurrent), self::MONTH_PERIOD)));
                    $recurrent = '+' . $qty . ' ' . self::MONTH_PERIOD;
                } elseif (strstr(mb_strtolower($recurrent), self::YEAR_PERIOD)) {
                    $qty = intval(substr($recurrent, 0, strpos(strtolower($recurrent), self::YEAR_PERIOD)));
                    $recurrent = '+' . $qty . ' ' . self::YEAR_PERIOD;
                }
            } else {
                $recurrent = null;
            }

            //affiliate block
            $enabledPlugins = Tools_Plugins_Tools::getEnabledPlugins();
            foreach ($enabledPlugins as $plugin) {
                if ($plugin->getName() == 'seosambaaffiliatenetwork') {
                    $affiliate = Tools_Factory_PluginFactory::createPlugin('seosambaaffiliatenetwork');
                    $affiliateStatus = $affiliate->affiliateSave($cartId, 'stripepay', 'creditcard');
                }
            }
            //affiliate end

            if (!empty($paymentMethod = $data['paymentMethod'])) {
                try {
                    $gw = new Tools_StripeRecurrent($privateKey, $paymentMethod, $paymentData);
                    $payResult = $gw->processPaymentIntent(self::PAYMENT_TYPE_SINGLE_PRODUCT);
                } catch (Exception $e) {
                    $payResult['error'] = 1;
                    $payResult['message'] = $this->_translator->translate(self::ERROR_PAYMENT_COMMON);
                }
            } else {
                $payResult['error'] = 1;
                $payResult['message'] = $this->_translator->translate(self::ERROR_PAYMENT_METHOD_NOT_FOUND);
            }
            if (empty($payResult)) {
                $payResult['error'] = 1;
                $payResult['message'] = $this->_translator->translate(self::ERROR_PAYMENT_CONNECTION_ERROR);
                $this->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_ERROR, false, $payResult['message']);
                echo json_encode($payResult);
                return;
            } elseif ($payResult['error'] == 0) {
                $this->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_COMPLETED);
                if (!empty($recurrent) && isset($payResult['paymentMethodId'])) {
                    $subscriptionId = $cartId . '_@@_' . $payResult['paymentMethodId'];
                    Tools_RecurringPaymentTools::createRecurringPaymentInfo($cartId, 65535, $subscriptionId, '',
                        $recurrent, $amount, 'stripepay', $amount, '', self::ACTIVE_STATUS, 1, 1);
                }
                $this->_sessionHelper->storeCartSessionKey = $cartId;
                $this->_sessionHelper->storeCartSessionConversionKey = $cartId;
            } else {
                $this->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_ERROR, false, isset($payResult['message']) ? $payResult['message'] : $this->_translator->translate(self::ERROR_PAYMENT_COMMON));
            }
            echo json_encode($payResult);
            return;
        }
    }

    /**
     * Create new customer address
     *
     * @param int $customerId user Id
     * @param array $address
     * @param null $type
     * @return mixed|null
     */
    private function _addAddress($customerId, $address, $type = null)
    {
        $addressTable = new Models_DbTable_CustomerAddress();
        if (!empty($address)) {
            if ($type !== null) {
                $address['address_type'] = $type;
            }
            $address = Tools_Misc::clenupAddress($address);
            $address['id'] = Tools_Misc::getAddressUniqKey($address);
            $address['user_id'] = $customerId;
            if (null === ($row = $addressTable->find($address['id'])->current())) {
                $row = $addressTable->createRow();
            }
            $row->setFromArray($address);

            return $row->save();
        }
        return null;
    }

    /**
     *  get shipping as billing info
     */
    public function shippingAsBillingAction()
    {
        $shippingToBilling = $this->_request->getParam('shippingToBilling');
        if ($shippingToBilling == '1') {
            $shippingAddressKey = Tools_ShoppingCart::getInstance()->getShippingAddressKey();
            if ($shippingAddressKey != null) {
                $paymentParams = Tools_ShoppingCart::getAddressById($shippingAddressKey);
                echo json_encode(array('error' => '0', 'shipping' => $paymentParams));
            }
        } else {
            if (Models_Mapper_ShoppingConfig::getInstance()->getConfigParam('country') != '') {
                $predefinedCountry = Models_Mapper_ShoppingConfig::getInstance()->getConfigParam('country');
            } else {
                $predefinedCountry = 'US';
            }

            echo json_encode(array('error' => '1', 'shippingPredefinedCountry' => $predefinedCountry));
        }
    }

    public function buttonPaymentSuccessCallbackAction()
    {
        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();
        if(empty($stripepayOAuth)) {
            $stripeConfig = $this->_getStripeConfig();
            $privateKey = $stripeConfig['privateKey'];
            $endpoint_secret = $stripeConfig['endpointSecret'];

            Stripe::setApiKey($privateKey);
            $payload = @file_get_contents('php://input');
            $sig_header = $_SERVER['HTTP_STRIPE_SIGNATURE'];
            $event = null;
            try {
                $event = Webhook::constructEvent(
                    $payload, $sig_header, $endpoint_secret
                );
            } catch (UnexpectedValueException $e) {
                // Invalid payload
                http_response_code(400);
                exit();
            } catch (Exception\SignatureVerificationException $e) {
                // Invalid signature
                http_response_code(400);
                exit();
            }
            // Handle the checkout.session.completed or charge.succeeded event
            if ($event->type == 'checkout.session.completed') {
                $session = $event->data->object;

                // Fulfill the purchase...
                $this->_handleCheckoutSession($session);
            } elseif ($event->type == 'charge.succeeded') {
                $session = $event->data->object;
                //error_log(print_r($session, true));

                $pluginSeosambapos = Application_Model_Mappers_PluginMapper::getInstance()->findByName('seosambapos');
                if ($pluginSeosambapos instanceof Application_Model_Models_Plugin && $pluginSeosambapos->getStatus() === Application_Model_Models_Plugin::ENABLED) {
                    // Fulfill the purchase...
                    $result = Tools_CartTools::completePurchase($session);

                    if($result['error']) {
                        http_response_code(400);
                        exit();
                    } else {
                        http_response_code(200);
                        exit();
                    }
                }
            }

            http_response_code(200);
            exit();
        } else {
            http_response_code(202);//Accepted
            exit();
        }
    }

    public static function mojoSessionEventSuccessPurchase($data)
    {
//        $data = array(
//            'eventType' => 'checkout.session.completed',//charge.succeeded
//            'session' => 'jsonObj',
//        );

//        $data = array(
//            'eventType' => 'charge.succeeded',
//            'session' => '{"id":"ch_3SIo60Jr9ie4OscB0IBHdry6","object":"charge","amount":1500,"amount_captured":1500,"amount_refunded":0,"application":"ca_T1XCTjzI73AEEdWW9IWuyFYNeJdeHnwl","application_fee":null,"application_fee_amount":null,"authorization_code":"976774","balance_transaction":"txn_3SIo60Jr9ie4OscB0Ymu0EGN","billing_details":{"address":{"city":null,"country":null,"line1":null,"line2":null,"postal_code":null,"state":null},"email":null,"name":null,"phone":null,"tax_id":null},"calculated_statement_descriptor":"SAN-FRANCISCO-BIKES-CL","captured":true,"created":1760609284,"currency":"gbp","customer":null,"description":null,"destination":null,"dispute":null,"disputed":false,"failure_balance_transaction":null,"failure_code":null,"failure_message":null,"fraud_details":[],"livemode":false,"metadata":{"type":"mobile","cartId":"8","websiteId":"500035"},"on_behalf_of":null,"order":null,"outcome":{"advice_code":null,"network_advice_code":null,"network_decline_code":null,"network_status":"approved_by_network","reason":null,"risk_level":"not_assessed","seller_message":"Payment complete.","type":"authorized"},"paid":true,"payment_intent":"pi_3SIo60Jr9ie4OscB0vzeeOrg","payment_method":"pm_1SIo6KJr9ie4OscBEU65AbXV","payment_method_details":{"card_present":{"amount_authorized":1500,"brand":"visa","brand_product":null,"cardholder_name":"CARDHOLDER\/VISA","country":"US","description":"Visa Purchasing","emv_auth_data":"8A023030","exp_month":3,"exp_year":2023,"fingerprint":"MUTVwK1GKveQ5yqL","funding":"credit","generated_card":null,"incremental_authorization_supported":false,"issuer":"Celtic Bank","last4":"9969","network":"visa","network_transaction_id":"801191226890707","offline":{"stored_at":null,"type":null},"overcapture_supported":false,"preferred_locales":["en"],"read_method":"contactless_emv","receipt":{"account_type":"credit","application_cryptogram":"FCA63314F6AEF8CA","application_preferred_name":"Stripe Credit","authorization_code":"976774","authorization_response_code":"3030","cardholder_verification_method":"approval","dedicated_file_name":"A000000003101001","terminal_verification_results":"0000000000","transaction_status_information":"0000"}},"type":"card_present"},"radar_options":[],"receipt_email":null,"receipt_number":null,"receipt_url":"https:\/\/pay.stripe.com\/receipts\/payment\/CAcaFwoVYWNjdF8xUEVSa2VKcjlpZTRPc2NCKIWIw8cGMgazhrTO-7M6LBbh9r-2byu6N1sKfLhRoj_5vvEP7pBq54-ljr9dVo7ut0PrKBT-JrhdDVxO","refunded":false,"review":null,"shipping":null,"source":null,"source_transfer":null,"statement_descriptor":null,"statement_descriptor_suffix":null,"status":"succeeded","transfer_data":null,"transfer_group":null}'
//        );


        $translator = Zend_Registry::get('Zend_Translate');
        $stripepayOAuth = Tools_StripeOAuthTools::checkStripepayConfigOAuth();
        if(!empty($stripepayOAuth)) {
            if(!empty($data['eventType']) && !empty($data['session'])) {
                $session = json_decode($data['session'], true);

                if(!empty($session) && is_array($session)) {
                    $session = (object) $session;
                }

                if($data['eventType'] == 'checkout.session.completed') {//stripepay
                    Stripepay::_handleCheckoutSession($session);

                    return array('error' => '0', 'code' => '200', 'message' => $translator->translate('Success.'));
                    //http_response_code(200);
                    //exit();
                } elseif ($data['eventType'] == 'charge.succeeded') {//seosambapos
                    $pluginSeosambapos = Application_Model_Mappers_PluginMapper::getInstance()->findByName('seosambapos');
                    if ($pluginSeosambapos instanceof Application_Model_Models_Plugin && $pluginSeosambapos->getStatus() === Application_Model_Models_Plugin::ENABLED) {
                        $result = Tools_CartTools::completePurchase($session);

                        if($result['error']) {
                            return array('error' => '1', 'code' => '400', 'message' => $result['message']);
                            //http_response_code(400);
                            //exit();
                        } elseif (empty($result['error']) && $result['accepted']) {
                            return array('error' => '0', 'code' => '202', 'message' => $translator->translate('Accepted.'));
                            //http_response_code(202);//Accepted
                            //exit();
                        } else {
                            return array('error' => '0', 'code' => '200', 'message' => $translator->translate('Success.'));
                            //http_response_code(200);
                            //exit();
                        }
                    } else {
                        return array('error' => '0', 'code' => '202', 'message' => $translator->translate('Accepted.'));
                        //http_response_code(202);//Accepted
                        //exit();
                    }
                } else {
                    return array('error' => '0', 'code' => '202', 'message' => $translator->translate('Accepted.'));
                    //http_response_code(202);//Accepted
                    //exit();
                }
            }
        } else {
            return array('error' => '0', 'code' => '202', 'message' => $translator->translate('Accepted.'));
            //http_response_code(202);//Accepted
            //exit();
        }
    }

    private function _handleCheckoutSession($session)
    {
        if (!empty($session) && isset($session->id)) {
            $stripeTransactionMapper = Stripepay_Models_Mappers_StripepayTransactionMapper::getInstance();
            $stripeTransactionModel = $stripeTransactionMapper->findByCheckoutSessionId($session->id);
            if ($stripeTransactionModel instanceof Stripepay_Models_Models_StripepayTransactionModel) {
                try {
                    $sessionCustomer = '';
                    if(!is_null($session->customer)) {
                        $sessionCustomer = $session->customer;
                    }

                    $stripeTransactionModel->setStatus(self::TRANSACTION_STATUS_SUCCESS);
                    $stripeTransactionModel->setStripeCustomerId($sessionCustomer);
                    $stripeTransactionModel->setPaymentIntent($session->payment_intent);
                    $stripeTransactionModel->setEmailSent('1');
                    $stripeTransactionModel->setOrderDateFull(Tools_System_Tools::convertDateFromTimezone('now'));
                    $stripeTransactionMapper->save($stripeTransactionModel);

                    $cartId = $stripeTransactionModel->getCartId();
                    if (!empty($cartId)) {
                        //affiliate block
                        $enabledPlugins = Tools_Plugins_Tools::getEnabledPlugins();
                        foreach ($enabledPlugins as $plugin) {
                            if ($plugin->getName() == 'seosambaaffiliatenetwork') {
                                $affiliate = Tools_Factory_PluginFactory::createPlugin('seosambaaffiliatenetwork');
                                $affiliateStatus = $affiliate->affiliateSave($cartId, 'stripepay', 'button');
                            }
                        }
                        //affiliate end
                        $stripePay = new Stripepay(array(), array());
                        $stripePay->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_COMPLETED);
                        $sessionHelper = Zend_Controller_Action_HelperBroker::getStaticHelper('session');
                        $sessionHelper->storeCartSessionKey = $cartId;
                        $sessionHelper->storeCartSessionConversionKey = $cartId;
                    }
                } catch (Exception $e) {
                    error_log($e->getMessage());
                }
            } else {
                error_log($this->_translator->translate('Cannot find the transaction with checkoutSessionId = ') . '"' . $session->id . '"');
            }
        }
    }

    /**
     * Method for CRON recurring payments
     * This method is receiving a recurring cart id and processing it.
     */
    public function recurrentPayAction()
    {
        $stripeConfig = $this->_getStripeConfig();
        $privateKey = $stripeConfig['privateKey'];
        $cartId = filter_var($this->_request->getParam('cartId'), FILTER_SANITIZE_STRING);
        if (!empty($cartId)) {
            $recurringPaymentMapper = Store_Mapper_RecurringPaymentsMapper::getInstance();
            $recurringPayments = $recurringPaymentMapper->getByCartId($cartId);
            $cartSessionMapper = Models_Mapper_CartSessionMapper::getInstance();

            if (!empty($recurringPayments) && is_array($recurringPayments)) {
                foreach ($recurringPayments as $recurrent) {

                    if (!in_array($recurrent->getRecurringStatus(), [Models_Model_CartSession::CART_STATUS_PENDING, Models_Model_CartSession::CART_STATUS_NEW, self::ACTIVE_STATUS])) {
                        $this->_responseHelper->fail($this->_translator->translate('The subscription status is') . " '" . $recurrent->getRecurringStatus() . "'");
                    }
                    if ($recurrent->getNextPaymentDate() > date('Y-m-d')) {
                        $this->_responseHelper->success($this->_translator->translate('Today no recurrence for this cart'));
                    }
                    $dependentCart = $cartSessionMapper->find($cartId);
                    $user = Application_Model_Mappers_UserMapper::getInstance()->find($dependentCart->getUserId());
                    $recurrent->setRecurringStatus(Models_Model_CartSession::CART_STATUS_PENDING);
                    $freeTransCycle = $recurrent->getFreeTransactionCycle();
                    if (!empty($freeTransCycle)) {
                        $transQuantity = $recurrent->getTransactionsQuantity();
                        $freeCart = ($transQuantity + 1) % $recurrent->getFreeTransactionCycle();
                        if (!$freeCart) {
                            $dependentCart->setFreeCart(1);
                            $dependentCart->setId(null);
                            $cartId = $cartSessionMapper->save($dependentCart)->getId();
                            Tools_RecurringPaymentTools::updateRecurringPaymentInfo($recurrent->getCartId(), self::ACTIVE_STATUS,
                                'stripepay', 0, $cartId);
                            $this->_responseHelper->success($this->_translator->translate('Success: 100'));
                        }
                    }
                    $recurringPaymentMapper->save($recurrent);

                    $subscriptionId = $recurrent->getSubscriptionId();
                    $parentCartId = substr($subscriptionId, 0, strpos($subscriptionId, '_@@_'));
                    $stripeTransactionMapper = Stripepay_Models_Mappers_StripepayTransactionMapper::getInstance();
                    $stripeTransaction = $stripeTransactionMapper->findByCartId($parentCartId, self::PAYMENT_TYPE_CART, array(Tools_StripeRecurrent::PAYMENT_INTENT_STATUS_SUCCESS, Models_Model_CartSession::CART_STATUS_COMPLETED));
                    if (empty($stripeTransaction)) {
                        $stripeTransaction = $stripeTransactionMapper->findByCartId($parentCartId, self::PAYMENT_TYPE_SINGLE_PRODUCT, array(Tools_StripeRecurrent::PAYMENT_INTENT_STATUS_SUCCESS, Models_Model_CartSession::CART_STATUS_COMPLETED));
                    }
                    if (empty($stripeTransaction)) {
                        $stripeTransaction = $stripeTransactionMapper->findByCartId($parentCartId, self::PAYMENT_TYPE_CART_QUOTE, array(Tools_StripeRecurrent::PAYMENT_INTENT_STATUS_SUCCESS, Models_Model_CartSession::CART_STATUS_COMPLETED));
                    }
                    if (isset($stripeTransaction[0]) && $stripeTransaction[0] instanceof Stripepay_Models_Models_StripepayTransactionModel) {
                        $stripeTransaction = $stripeTransaction[0];
                        $dependentCart->setId(null);
                        $dependentCart->setStatus(Models_Model_CartSession::CART_STATUS_PROCESSING);
                        $cartId = $cartSessionMapper->save($dependentCart)->getId();
                        $paymentData = [];
                        $paymentData['amount'] = $dependentCart->getTotal();
                        $paymentData['totalTax'] = $dependentCart->getTotalTax();
                        $paymentData['cartId'] = $cartId;
                        $paymentData['customerShipping'] = $stripeTransaction->getOrderShippingAddress();
                        $paymentData['billingAddress'] = $stripeTransaction->getOrderBillingAddress();
                        $paymentData['userFirstName'] = $stripeTransaction->getFirstName();
                        $paymentData['userLastName'] = $stripeTransaction->getLastName();
                        $paymentData['userEmail'] = $stripeTransaction->getEmail();
                        $paymentData['dependentCart'] = $dependentCart;
                        try {
                            $gw = new Tools_StripeRecurrent($privateKey, $stripeTransaction->getPaymentMethod(), $paymentData);
                            $payResult = $gw->processPaymentIntent(self::PAYMENT_TYPE_RECURRENT, $stripeTransaction->getStripeCustomerId());
                        } catch (Exception $e) {
                            $payResult['error'] = 1;
                            $payResult['message'] = $this->_translator->translate(self::ERROR_PAYMENT_COMMON);
                            $this->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_ERROR, false, $payResult['message']);
                        }
                    }
                    if (empty($payResult)) {
                        $payResult['error'] = 1;
                        $payResult['message'] = $this->_translator->translate(self::ERROR_PAYMENT_CONNECTION_ERROR);
                        $this->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_ERROR, false, $payResult['message']);
                        $this->_responseHelper->fail('Error: ' . $payResult['message']);
                    } elseif ($payResult['error'] == 0) {
                        Tools_RecurringPaymentTools::updateRecurringPaymentInfo($recurrent->getCartId(), self::ACTIVE_STATUS,
                            'stripepay', $paymentData['amount'], $cartId);
                        $this->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_COMPLETED);

                        $pluginInvoicePdf = Application_Model_Mappers_PluginMapper::getInstance()->findByName('invoicetopdf');
                        if ($pluginInvoicePdf instanceof Application_Model_Models_Plugin && $pluginInvoicePdf->getStatus() === Application_Model_Models_Plugin::ENABLED) {
                            $invoicetopdfConfig = Invoicetopdf_Models_Mapper_InvoicetopdfSettingsMapper::getInstance()->getConfigParams('attachInvoiceActionEmail');
                            if (isset($invoicetopdfConfig['attachInvoiceActionEmail']) && $invoicetopdfConfig['attachInvoiceActionEmail'] === '1') {
                                $fileInfo = Tools_Misc::prepareInvoice(['cartId' => $recurrent->getCartId(), 'dwn' => 0]);
                                if (!empty($fileInfo)) {
                                    $recurrent->registerObserver(new Tools_Mail_Watchdog(array(
                                        'trigger' => Tools_StripepayMailWatchdog::TRIGGER_SUBSCRIPTION_RECURRING_PAYMENT,
                                        'fullName' => $user->getFullName(),
                                        'userEmail' => $user->getEmail(),
                                        'userId' => $dependentCart->getUserId(),
                                        'attachmentUrl' => $this->_websiteHelper->getUrl() . 'plugins/invoicetopdf/invoices/' . $fileInfo['fileName'],
                                        'attachmentPath' => $this->_pdfPath . $fileInfo['fileName'],
                                        'attachmentName' => $fileInfo['fileName']
                                    )));

                                }
                            } else {
                                $recurrent->registerObserver(new Tools_Mail_Watchdog(array(
                                    'trigger' => Tools_StripepayMailWatchdog::TRIGGER_SUBSCRIPTION_RECURRING_PAYMENT,
                                    'fullName' => $user->getFullName(),
                                    'userEmail' => $user->getEmail(),
                                    'userId' => $dependentCart->getUserId(),
                                )));
                            }
                            $recurrent->notifyObservers();
                        }
                    } else {
                        $this->updateCartStatus($cartId, Models_Model_CartSession::CART_STATUS_ERROR, false, isset($payResult['message']) ? $payResult['message'] : $this->_translator->translate(self::ERROR_PAYMENT_COMMON));
                        $recurrent->registerObserver(new Tools_Mail_Watchdog(array(
                            'trigger' => Tools_StripepayMailWatchdog::TRIGGER_SUBSCRIPTION_RECURRING_FAILED,
                            'fullName' => $user->getFullName(),
                            'userEmail' => $user->getEmail(),
                            'userId' => $dependentCart->getUserId()
                        )));
                        $recurrent->notifyObservers();
                        $this->_responseHelper->fail('Error: ' . isset($payResult['message']) ? $payResult['message'] : $this->_translator->translate(self::ERROR_PAYMENT_COMMON));
                    }
                }
                $this->_responseHelper->success($this->_translator->translate('Success: 100'));
            }
            $this->_responseHelper->success($this->_translator->translate('Success: No recurrence for this cart'));
        }
        $this->_responseHelper->fail();
    }

    public function refund($orderId, $refundAmount, $refundNotes)
    {
        $payResult = array('error' => 1);
        $stripeConfig = $this->_getStripeConfig();
        $privateKey = $stripeConfig['privateKey'];
        $transactionMapper = Stripepay_Models_Mappers_StripepayTransactionMapper::getInstance();
        $transForRefund = $transactionMapper->findByCartId($orderId, false, array(self::TRANSACTION_STATUS_SUCCESS, Models_Model_CartSession::CART_STATUS_COMPLETED));
        if (!isset($transForRefund[0]) || !($transForRefund[0] instanceof Stripepay_Models_Models_StripepayTransactionModel)) {
            $payResult['errorMessage'] = $this->_translator->translate("There is no order with such ID");
            return $payResult;
        }
        $transForRefund = $transForRefund[0];
        $paymentIntent = $transForRefund->getPaymentIntent();
        if (!$paymentIntent) {
            $chargeId = $transForRefund->getChargeId();
            if (!$chargeId) {
                $payResult['errorMessage'] = $this->_translator->translate("The transaction cannot be refunded.");
                return $payResult;
            }
            $refund = Tools_RecurringPaymentStripepay::refundCharge($privateKey, $chargeId, round($refundAmount, 2) * 100);
        } else {
            $refund = Tools_RecurringPaymentStripepay::refundPaymentIntent($privateKey, $paymentIntent, round($refundAmount, 2) * 100);
        }
        if (!is_null($refund) && is_object($refund) && $refund->status == Tools_StripeRecurrent::PAYMENT_INTENT_STATUS_SUCCESS) {
            $prevRefundAmount = $transForRefund->getRefundAmount() ? $transForRefund->getRefundAmount() : 0;
            $transForRefund->setStatus(Models_Model_CartSession::CART_STATUS_REFUNDED);
            $transForRefund->setRefundId($refund->id);
            $transForRefund->setRefundAmount($prevRefundAmount + $refundAmount);
            $transForRefund->setRefundLastDate(Tools_System_Tools::convertDateFromTimezone('now'));
            $transForRefund->setReason($refundNotes);
            $transactionMapper->save($transForRefund);
            $payResult['error'] = 0;
            $payResult['refundMessage'] = $this->_translator->translate("The transaction has been successfully refunded.");
        } elseif (isset($refund['errorMessage'])) {
            $payResult['errorMessage'] = $refund['errorMessage'];
        } else {
            $payResult['errorMessage'] = self::ERROR_PAYMENT_COMMON;
        }

        return $payResult;
    }

    protected function _setRecalculationFlags($id = false) {
        $recalculationFlags = [
            'skipGroupPriceRecalculation' => false,
            'skipOptionRecalculation' => false
        ];
        if ($id) {
            $userModel = Application_Model_Mappers_UserMapper::getInstance()->find($id);
            if ($userModel instanceof Application_Model_Models_User) {
                $userRoleId = $userModel->getRoleId();
                if ($userRoleId == Shopping::ROLE_SALESPERSON || $userRoleId == Tools_Security_Acl::ROLE_ADMIN || $userRoleId == Tools_Security_Acl::ROLE_SUPERADMIN) {
                    $recalculationFlags['skipGroupPriceRecalculation'] = true;
                    $recalculationFlags['skipOptionRecalculation'] = true;
                }
            }
        }
        return $recalculationFlags;
    }

    /**
     * Generate product unique key in cart
     *
     * @param Models_Model_Product $item
     * @param array $options
     * @return string
     */
    private function _generateStorageKey($item, $options = array())
    {
        return substr(md5($item->getName() . $item->getSku() . http_build_query($options)), 0, 10);
    }

    /**
     *
     * Prepare product options
     *
     * @param array $options
     * @return array
     */
    protected function _parseProductOptions($options)
    {
        $parsedOption = array();
        foreach ($options as $option) {
            $parsedOption[$option['id']] = $option['option_id'];
        }
        return $parsedOption;
    }

    public function deauthorizeAccountAction()
    {
        if (Tools_Security_Acl::isAllowed(Tools_Security_Acl::RESOURCE_PLUGINS) && $this->_request->isPost()) {
            $tokenToValidate = $this->_request->getParam(Tools_System_Tools::CSRF_SECURE_TOKEN, false);
            $valid = Tools_System_Tools::validateToken($tokenToValidate, self::STRIPE_SECURE_TOKEN);
            if (!$valid) {
                $this->_responseHelper->fail('');
            }

            $status = Tools_StripeOAuthTools::deauthorizeAccountMojo();

            if(!empty($status)) {
                $this->_responseHelper->success(array('status' => 'disconnected'));
            }
        }
        $this->_responseHelper->fail($this->_translator->translate('Can not deauthorize account.'));
    }

}
