<?php

/**
 * Class Tools_RecurringPaymentPaypal
 */

class Tools_RecurringPaymentPaypal implements Interfaces_RecurringPayment {

    const API_VERSION = '95.0';

    const LIVE_API_GATEWAY_URL = 'https://api-3t.paypal.com/nvp';

    const SANDBOX_API_GATEWAY_URL = 'https://api-3t.sandbox.paypal.com/nvp';

    const RECURRING_PROFILE_STATUS_ACTIVE = 'Active';

    const RECURRING_PROFILE_STATUS_SUSPENDED = 'Suspended';

    const RECURRING_PROFILE_STATUS_CANCELED = 'Canceled';

    /**
     * Only profiles in Active or Suspended state can be canceled.
     */
    const RECURRING_PROFILE_ACTION_CANCEL = 'Cancel';

    /**
     * Only profiles in Active state can be suspended.
     */
    const RECURRING_PROFILE_ACTION_SUSPEND = 'Suspend';

    /**
     *  Only profiles in a suspended state can be reactivated.
     */
    const RECURRING_PROFILE_ACTION_REACTIVATE = 'Reactivate';


    /**
     * Creates a recurring payments profile. You must invoke the CreateRecurringPaymentsProfile API operation for each
     * profile you want to create. The API operation creates a profile and an associated billing agreement.
     *
     * @see https://developer.paypal.com/docs/classic/api/merchant/CreateRecurringPaymentsProfile_API_Operation_NVP/
     */
    const CREATE_RECURRING_PAYMENTS_PROFILE = 'CreateRecurringPaymentsProfile';

    /**
     * Processes a payment from a buyer's account, which is identified by a previous transaction.
     *
     * @see https://developer.paypal.com/docs/classic/api/merchant/DoReferenceTransaction_API_Operation_NVP/
     */
    const DO_REFERENCE_TRANSACTION = 'DoReferenceTransaction';

    /**
     * Shows information about a recurring payments profile.
     *
     * @see https://developer.paypal.com/docs/classic/api/merchant/GetRecurringPaymentsProfileDetails_API_Operation_NVP/
     */
    const GET_RECURRING_PAYMENTS_PROFILE_DETAILS = 'GetRecurringPaymentsProfileDetails';

    /**
     * The ManageRecurringPaymentsProfileStatus API operation cancels, suspends, or reactivates a recurring
     * payments profile.
     *
     * @see https://developer.paypal.com/docs/classic/api/merchant/ManageRecurringPaymentsProfileStatus_API_Operation_NVP/
     */
    const MANAGE_RECURRING_PROFILE_STATUS  = 'ManageRecurringPaymentsProfileStatus';

    /**
     * Updates a recurring payments profile.
     *
     * @see https://developer.paypal.com/docs/classic/api/merchant/UpdateRecurringPaymentsProfile_API_Operation_NVP/
     */
    const UPDATE_RECURRING_PAYMENTS_PROFILE = 'UpdateRecurringPaymentsProfile';

    /**
     * Bills the buyer for the outstanding balance associated with a recurring payments profile.
     *
     * @see https://developer.paypal.com/docs/classic/api/merchant/BillOutstandingAmount_API_Operation_NVP/
     */
    const BILL_OUTSTANDING_AMOUNT = 'BillOutstandingAmount';

    /**
     * Profile Created
     */
    const RECURRING_PROFILE_TYPE_CREATED = 'recurring_payment_profile_created';

    /**
     * Regular
     */
    const RECURRING_PROFILE_TYPE_REGULAR = 'recurring_payment';

    /**
     * Skipped
     */
    const RECURRING_PROFILE_TYPE_SKIPPED = 'recurring_payment_skipped';

    /**
     * Failed
     */
    const RECURRING_PROFILE_TYPE_FAILED = 'recurring_payment_failed';

    /**
     * Suspended
     */
    const RECURRING_PROFILE_TYPE_SUSPENDED_DUE_TO_MAX = 'recurring_payment_suspended_due_to_max_failed_payment';

    /**
     * Suspended
     */
    const RECURRING_PROFILE_TYPE_SUSPENDED = 'recurring_payment_suspended';

    /**
     * Cancel
     */
    const RECURRING_PROFILE_TYPE_CANCEL = 'recurring_payment_profile_cancel';

    /**
     * Recurring profile types
     *
     * @var array
     */
    public static $_recurringProfileTypes = array(
        self::RECURRING_PROFILE_TYPE_CREATED,
        self::RECURRING_PROFILE_TYPE_REGULAR,
        self::RECURRING_PROFILE_TYPE_SKIPPED,
        self::RECURRING_PROFILE_TYPE_FAILED,
        self::RECURRING_PROFILE_TYPE_SUSPENDED_DUE_TO_MAX,
        self::RECURRING_PROFILE_TYPE_SUSPENDED,
        self::RECURRING_PROFILE_TYPE_CANCEL
    );

    protected $_cartId;

    protected $_recurringPaymentData;

    protected $_translator;

    public function __construct($data = array())
    {
        if (isset($data['cartId'])) {
            $this->_cartId = $data['cartId'];
        }

        $this->_translator = Zend_Registry::get('Zend_Translate');
        $this->_recurringPaymentData = $data;

    }

    public function updateRecurringPayment()
    {
        $recurrentCartId = $this->_cartId;
        if (empty($recurrentCartId)) {
            return array('error' => true, 'message' => 'Missing cartId');
        }
        $recurringPaymentsMapper = Store_Mapper_RecurringPaymentsMapper::getInstance();
        $recurringPayment = $recurringPaymentsMapper->getByCartId($recurrentCartId);
        if (!empty($recurringPayment) && $recurringPayment[0] instanceof Store_Model_RecurringPayments) {
            $recurringPayment = $recurringPayment[0];
            $profileStatus = $recurringPayment->getRecurringStatus();
            if ($profileStatus == Store_Model_RecurringPayments::CANCELED_RECURRING_PAYMENT) {
                return array('error' => true, 'message' => $this->_translator->translate('The subscription cannot be reactivated'));
            }

            if ($profileStatus !== Store_Model_RecurringPayments::SUSPENDED_RECURRING_PAYMENT) {
                return array('error' => true, 'message' => $this->_translator->translate('Sorry. Paypal does not allow to change payment period.'));
            }

            if (!empty($this->_recurringPaymentData['paymentCycle'])) {
                $paymentCycle = strtolower($this->_recurringPaymentData['paymentCycle']);
                if (!empty($paymentCycle)) {

                    if ($paymentCycle == 'day') {
                        $period = '+1 day';
                    } elseif ($paymentCycle == 'week') {
                        $period = '+1 week';
                    } elseif ($paymentCycle == 'month') {
                        $period = '+1 month';
                    } elseif ($paymentCycle == 'month-two') {
                        $period = '+2 month';
                    } elseif ($paymentCycle == 'quarter') {
                        $period = '+3 month';
                    } elseif ($paymentCycle == 'semester') {
                        $period = '+6 month';
                    } elseif ($paymentCycle == 'year') {
                        $period = '+1 year';
                    } else {
                        return array('error' => true, 'message' => $this->_translator->translate('Paypal does not allow use payment period' . ' "' . $this->_recurringPaymentData['paymentCycle'] . '"'));
                    }

                    if (!empty($period)) {
                        $nextPaymentDate = date('Y-m-d', strtotime($period));
                        $recurringPayment->setNextPaymentDate($nextPaymentDate);
                        $recurringPayment->setPaymentPeriod(strtoupper($period));
                    }
                }
            }

            $profileId = $recurringPayment->getSubscriptionId();
            if ($profileStatus === Store_Model_RecurringPayments::SUSPENDED_RECURRING_PAYMENT) {
                $profileReactivate = self::reactivateRecurringPaymentProfile($profileId);
                if ($profileReactivate['error']) {
                    return $profileReactivate;
                }
                $recurringPayment->setRecurringStatus(Store_Model_RecurringPayments::ACTIVE_RECURRING_PAYMENT);
            }

            $recurringPaymentsMapper->save($recurringPayment);
            $recurrentCart = Models_Mapper_CartSessionMapper::getInstance()->find($recurrentCartId);
            $user = Application_Model_Mappers_UserMapper::getInstance()->find($recurrentCart->getUserId());
            $updateLocalProfileResponse = Tools_RecurringPaymentPaypal::updateLocalProfileInfo($profileId, $recurrentCart->getUserId(), $recurrentCartId);
            $recurringPayment->registerObserver(new Tools_Mail_Watchdog(array(
                'trigger' => Tools_PaypalMailWatchdog::TRIGGER_SUBSCRIPTION_RECURRING_REACTIVATED,
                'fullName' => $user->getFullName(),
                'userEmail' => $user->getEmail(),
                'userId' => $recurrentCart->getUserId()
            )));
            $recurringPayment->notifyObservers();
        } else {
            return array('error' => true, 'message' => 'Recurring profile not found.');
        }

        return array('error' => false, 'message' => 'Subscription reactivated');
    }

    public function suspendRecurringPayment()
    {
        $recurrentCartId = $this->_cartId;
        if (empty($recurrentCartId)) {
            return array('error' => true, 'message' => 'Missing cartId');
        }
        $recurringPaymentsMapper = Store_Mapper_RecurringPaymentsMapper::getInstance();
        $recurringPayment = $recurringPaymentsMapper->getByCartId($recurrentCartId);
        if (!empty($recurringPayment) && $recurringPayment[0] instanceof Store_Model_RecurringPayments) {
            $recurringPayment = $recurringPayment[0];
            $profileId = $recurringPayment->getSubscriptionId();
            $profileSuspend = self::suspendRecurringPaymentProfile($profileId);
            if ($profileSuspend['error']) {
                return $profileSuspend;
            }
            $recurringPayment->setRecurringStatus(Store_Model_RecurringPayments::SUSPENDED_RECURRING_PAYMENT);
            $recurringPaymentsMapper->save($recurringPayment);
            $recurrentCart = Models_Mapper_CartSessionMapper::getInstance()->find($recurrentCartId);
            $user = Application_Model_Mappers_UserMapper::getInstance()->find($recurrentCart->getUserId());
            $updateLocalProfileResponse = Tools_RecurringPaymentPaypal::updateLocalProfileInfo($profileId, $recurrentCart->getUserId(), $recurrentCartId);
        } else {
            return array('error' => true, 'message' => 'Recurring profile not found.');
        }

        return array('error' => false, 'message' => 'Recurring payment suspended');
    }

    public function cancelRecurringPayment()
    {
        $recurrentCartId = $this->_cartId;
        if (empty($recurrentCartId)) {
            return array('error' => true, 'message' => 'Missing cartId');
        }
        $recurringPaymentsMapper = Store_Mapper_RecurringPaymentsMapper::getInstance();
        $recurringPayment = $recurringPaymentsMapper->getByCartId($recurrentCartId);
        if (!empty($recurringPayment) && $recurringPayment[0] instanceof Store_Model_RecurringPayments) {
            $recurringPayment = $recurringPayment[0];
            $profileId = $recurringPayment->getSubscriptionId();
            $profileCanceled = self::cancelRecurringPaymentProfile($profileId);
            if ($profileCanceled['error']) {
                return $profileCanceled;
            } elseif (empty($profileCanceled)) {
                return array('error' => true, 'message' => 'Something went wrong.');
            }
            $recurringPayment->setRecurringStatus(Store_Model_RecurringPayments::CANCELED_RECURRING_PAYMENT);
            $recurringPaymentsMapper->save($recurringPayment);
            $recurrentCart = Models_Mapper_CartSessionMapper::getInstance()->find($recurrentCartId);
            $user = Application_Model_Mappers_UserMapper::getInstance()->find($recurrentCart->getUserId());
            $updateLocalProfileResponse = Tools_RecurringPaymentPaypal::updateLocalProfileInfo($profileId, $recurrentCart->getUserId(), $recurrentCartId);
        } else {
            return array('error' => true, 'message' => 'Recurring profile not found.');
        }

        return array('error' => false, 'message' => 'Recurring payment cancelled');
    }
    /**
     * Update recurring profile
     * @see https://developer.paypal.com/docs/classic/api/merchant/UpdateRecurringPaymentsProfile_API_Operation_NVP/
     *
     * @param string $profileId - Recurring payments profile ID returned in the CreateRecurringPaymentsProfile response.
     * @param string $creditCardNumber - Credit card number.
     * @param string $expirationDate - Credit card expiration date.
     * @param string $cvv - Card Verification Value, version 2.
     * @param string $email - Email address of buyer.
     * @param string $street - First street address. Character length and limitations: 100 single-byte characters
     * @param string $city - Name of city. 40 single-byte characters
     * @param string $state - State or province. 40 single-byte characters
     * @param string $countryCode - Country code. 2 single-byte characters
     * @param string $zip - U.S. ZIP code or other country-specific postal code. 20 single-byte characters
     * @param array $optionalBillingParams
     *      note - (Optional) The reason for the update to the recurring payments profile.
     *
     * @return array
     */
    public static function updateRecurringPaymentsProfile($profileId, $creditCardNumber, $cvv, $expirationDate,
        $email, $street, $city, $state, $countryCode, $zip, $optionalBillingParams = array())
    {
        $data = array();
        $data = self::addCredentials($data);
        if (!empty($data['error'])) {
            return array('error' => '1', $data['message']);
        }

        $data['METHOD'] = self::UPDATE_RECURRING_PAYMENTS_PROFILE;
        $data['PROFILEID'] = $profileId;
        $data['ACCT'] = $creditCardNumber;
        $data['CVV2'] = $cvv;
        $data['EXPDATE'] = $expirationDate;
        $data['EMAIL'] = $email;
        $data['STREET'] = $street;
        $data['CITY'] = $city;
        $data['STATE'] = $state;
        $data['COUNTRYCODE'] = $countryCode;
        $data['ZIP'] = $zip;


        if (!empty($optionalBillingParams['note'])) {
            $data['NOTE'] = $optionalBillingParams['note'];
        }

        $action = self::getEndpointUrl();

        $response = self::makeRequest('POST', $action, $data);
        if (!empty($response['L_ERRORCODE0'])) {
            return array('error' => '1', 'message' => $response['L_LONGMESSAGE0']);
        }

        if (empty($response)) {
            return array('error' => '1', 'message' => 'Payment processor outrage');
        }

        return $response;
    }


    /**
     * Create recurring profile
     *
     *
     * @see https://developer.paypal.com/docs/classic/api/merchant/CreateRecurringPaymentsProfile_API_Operation_NVP/
     *
     * @param string $profileStartDate - The date when billing for this profile begins.
     *      Must be a valid date, in UTC/GMT format; for example, 2013-08-24T05:38:48Z. No wildcards are allowed.
     * @param string $description - Description of the recurring payment.Character length and limitations:
     *     127 single-byte alphanumeric characters
     * @param string $billingPeriod - Unit for billing during this subscription period. Value is: Day, Week, SemiMonth, Month,
     * Year
     * @param string $billingFrequency - Number of billing periods that make up one billing cycle.
     *      The combination of billing frequency and billing period must be less than or equal to one year.
     * @param float $amt - billing amount for each billing cycle during this payment period. This amount does not include
     * shipping and tax amounts.
     * @param string $currencyCode - Currency code (default is USD). Character length and limitations: 3 single-byte characters
     * @param string $creditCardNumber - Credit card number.
     * @param string $expirationDate - Credit card expiration date.
     * @param string $cvv - Card Verification Value, version 2.
     * @param string $email - Email address of buyer.
     * @param string $street - First street address. Character length and limitations: 100 single-byte characters
     * @param string $city - Name of city. 40 single-byte characters
     * @param string $state - State or province. 40 single-byte characters
     * @param string $countryCode - Country code. 2 single-byte characters
     * @param string $zip - U.S. ZIP code or other country-specific postal code. 20 single-byte characters
     *
     * @param array $optionalBillingParams
     * subscriberName -  Full name of the person receiving the product or service paid for by the recurring payment.
     *      If not present, the name in the buyer's PayPal account is used.
     * profileReference - The merchant's own unique reference or invoice number.
     *      Character length and limitations: 127 single-byte alphanumeric characters
     * maxFailedPayments - Number of scheduled payments that can fail before the profile is automatically suspended.
     *      An IPN message is sent to the merchant when the specified number of failed payments is reached.
     * autoBillOutAtm - Indicates whether you would like PayPal to automatically bill the outstanding balance amount
     *      in the next billing cycle. The outstanding balance is the total amount of any previously failed scheduled
     * payments that have yet to be successfully paid. Value is: NoAutoBill — PayPal does not automatically bill the
     * outstanding balance. AddToNextBilling — PayPal automatically bills the outstanding balance.
     * totalBillingCycles - Number of billing cycles for payment period.
     * trialBillingPeriod - Unit for billing during this subscription period; required if you specify an optional trial
     * period. Value is: Day, Week, SemiMonth, Month, Year
     * trialBillingFrequency - Number of billing periods that make up one billing cycle; required if you specify an
     *      optional trial period.
     * trialToTotalBillingCycles - Number of billing cycles for trial payment period.
     * trialAmt - Billing amount for each billing cycle during this payment period;
     *      required if you specify an optional trial period. This amount does not include shipping and tax amounts.
     * shippingAmt - Shipping amount for each billing cycle during this payment period.
     * taxAmt - Tax amount for each billing cycle during this payment period.
     * initAmt -  Initial non-recurring payment amount due immediately upon profile creation.
     *      Use an initial amount for enrollment or setup fees.
     * failedInitAmtAction - Action you can specify when a payment fails ContinueOnFailure|CancelOnFailure
     * @return array
     */
    public static function createRecurringPaymentsProfile($profileStartDate, $description,
        $billingPeriod, $billingFrequency, $amt, $currencyCode, $creditCardNumber, $expirationDate, $cvv,
        $email, $street, $city, $state, $countryCode, $zip,
        $optionalBillingParams = array())
    {
        $data = array();
        $data = self::addCredentials($data);
        if (!empty($data['error'])) {
            return array('error' => '1', $data['message']);
        }

        if (!empty($optionalBillingParams['subscriberName'])) {
            $data['SUBSCRIBERNAME'] = $optionalBillingParams['subscriberName'];
        }
        if (!empty($optionalBillingParams['profileReference'])) {
            $data['PROFILEREFERENCE'] = $optionalBillingParams['profileReference'];
        }
        if (!empty($optionalBillingParams['maxFailedPayments'])) {
            $data['MAXFAILEDPAYMENTS'] = $optionalBillingParams['maxFailedPayments'];
        }
        if (!empty($optionalBillingParams['autoBillOutAtm'])) {
            $data['AUTOBILLOUTAMT'] = $optionalBillingParams['autoBillOutAtm'];
        }

        if (!empty($optionalBillingParams['totalBillingCycles'])) {
            $data['TOTALBILLINGCYCLES'] = $optionalBillingParams['totalBillingCycles'];
        }

        if (!empty($optionalBillingParams['trialBillingPeriod'])) {
            $data['TRIALBILLINGPERIOD'] = $optionalBillingParams['trialBillingPeriod'];
        }

        if (!empty($optionalBillingParams['trialBillingFrequency'])) {
            $data['TRIALBILLINGFREQUENCY'] = $optionalBillingParams['trialBillingFrequency'];
        }

        if (!empty($optionalBillingParams['trialToTotalBillingCycles'])) {
            $data['TRIALTOTALBILLINGCYCLES'] = $optionalBillingParams['trialToTotalBillingCycles'];
        }

        if (!empty($optionalBillingParams['trialAmt'])) {
            $data['TRIALAMT'] = $optionalBillingParams['trialAmt'];
        }

        if (!empty($optionalBillingParams['shippingAmt'])) {
            $data['SHIPPINGAMT'] = $optionalBillingParams['shippingAmt'];
        }

        if (!empty($optionalBillingParams['taxAmt'])) {
            $data['TAXAMT'] = $optionalBillingParams['taxAmt'];
        }

        if (!empty($optionalBillingParams['initAmt'])) {
            $data['INITAMT'] = $optionalBillingParams['initAmt'];
        }

        if (!empty($optionalBillingParams['failedInitAmtAction'])) {
            $data['FAILEDINITAMTACTION'] = $optionalBillingParams['failedInitAmtAction'];
        }

        $data['METHOD'] = self::CREATE_RECURRING_PAYMENTS_PROFILE;
        $data['PROFILESTARTDATE'] = $profileStartDate;
        $data['DESC'] = $description;
        $data['BILLINGPERIOD'] = $billingPeriod;
        $data['BILLINGFREQUENCY'] = $billingFrequency;
        $data['AMT'] = $amt;
        $data['CURRENCYCODE'] = $currencyCode;
        $data['ACCT'] = $creditCardNumber;
        $data['CVV2'] = $cvv;
        $data['EXPDATE'] = $expirationDate;
        $data['EMAIL'] = $email;
        $data['STREET'] = $street;
        $data['CITY'] = $city;
        $data['STATE'] = $state;
        $data['COUNTRYCODE'] = $countryCode;
        $data['ZIP'] = $zip;

        $action = self::getEndpointUrl();

        $response = self::makeRequest('POST', $action, $data);
        if (!empty($response['L_ERRORCODE0'])) {
            return array('error' => '1', 'message' => $response['L_LONGMESSAGE0']);
        }

        if (empty($response)) {
            return array('error' => '1', 'message' => 'Payment processor outrage');
        }

        if ($response['PROFILESTATUS']  === 'PendingProfile') {
            return array('error' => '1', 'message' => 'Pending profile error');
        }

        return $response;
    }

    /**
     * Get recurring payment profile details
     *
     * @see https://developer.paypal.com/docs/classic/api/merchant/GetRecurringPaymentsProfileDetails_API_Operation_NVP/
     *
     * @param string $profileId Recurring payments profile ID returned in the CreateRecurringPaymentsProfile response.
     * 19-character profile IDs are supported for compatibility with previous versions of the PayPal API.
     * @return array
     */
    public static function getRecurringPaymentsProfileDetails($profileId)
    {

        $data = array();
        $data = self::addCredentials($data);
        if (!empty($data['error'])) {
            return array('error' => '1', $data['message']);
        }

        $action = self::getEndpointUrl();
        $data['METHOD'] = self::GET_RECURRING_PAYMENTS_PROFILE_DETAILS;
        $data['PROFILEID'] = $profileId;

        $response = self::makeRequest('POST', $action, $data);

        if (!empty($response['L_ERRORCODE0'])) {
            return array('error' => '1', 'message' => $response['L_LONGMESSAGE0']);
        }

        return $response;

    }

    /**
     * Cancel recurring profile
     *
     * @see https://developer.paypal.com/docs/classic/api/merchant/ManageRecurringPaymentsProfileStatus_API_Operation_NVP/
     *
     * @param string $profileId Recurring payments profile ID returned in the CreateRecurringPaymentsProfile response.
     * 19-character profile IDs are supported for compatibility with previous versions of the PayPal API.
     * @return array
     */

    public static function cancelRecurringPaymentProfile($profileId)
    {
        return self::manageRecurringPaymentStatus($profileId, self::RECURRING_PROFILE_ACTION_CANCEL);
    }


    /**
     * suspend recurring profile
     *
     * @see https://developer.paypal.com/docs/classic/api/merchant/ManageRecurringPaymentsProfileStatus_API_Operation_NVP/
     *
     * @param string $profileId Recurring payments profile ID returned in the CreateRecurringPaymentsProfile response.
     * 19-character profile IDs are supported for compatibility with previous versions of the PayPal API.
     * @return array
     */

    public static function suspendRecurringPaymentProfile($profileId)
    {
        return self::manageRecurringPaymentStatus($profileId, self::RECURRING_PROFILE_ACTION_SUSPEND);
    }


    /**
     * reactivate recurring profile
     *
     * @see https://developer.paypal.com/docs/classic/api/merchant/ManageRecurringPaymentsProfileStatus_API_Operation_NVP/
     *
     * @param string $profileId Recurring payments profile ID returned in the CreateRecurringPaymentsProfile response.
     * 19-character profile IDs are supported for compatibility with previous versions of the PayPal API.
     * @return array
     */

    public static function reactivateRecurringPaymentProfile($profileId)
    {
        return self::manageRecurringPaymentStatus($profileId, self::RECURRING_PROFILE_ACTION_REACTIVATE);
    }

    /**
     * Manage recurring payment status
     *
     * @see https://developer.paypal.com/docs/classic/api/merchant/ManageRecurringPaymentsProfileStatus_API_Operation_NVP/
     *
     * @param string $profileId Recurring payments profile ID returned in the CreateRecurringPaymentsProfile response.
     * 19-character profile IDs are supported for compatibility with previous versions of the PayPal API.
     * @param string $status
     *      Cancel — Only profiles in Active or Suspended state can be canceled.
     *      Suspend — Only profiles in Active state can be suspended.
     *      Reactivate — Only profiles in a suspended state can be reactivated.
     * @return array
     */
    public static function manageRecurringPaymentStatus($profileId, $status)
    {
        $data = array();
        $data = self::addCredentials($data);
        if (!empty($data['error'])) {
            return array('error' => '1', $data['message']);
        }

        $action = self::getEndpointUrl();
        $data['METHOD'] = self::MANAGE_RECURRING_PROFILE_STATUS;
        $data['PROFILEID'] = $profileId;
        $data['ACTION'] = $status;

        $response = self::makeRequest('POST', $action, $data);

        if (!empty($response['L_ERRORCODE0'])) {
            return array('error' => '1', 'message' => $response['L_LONGMESSAGE0']);
        }

        return $response;

    }

    /**
     * Bill whole outstanding amount
     *
     * @see https://developer.paypal.com/docs/classic/api/merchant/BillOutstandingAmount_API_Operation_NVP/
     *
     * @param string $profileId Recurring payments profile ID returned in the CreateRecurringPaymentsProfile response.
     * 19-character profile IDs are supported for compatibility with previous versions of the PayPal API.
     * @return array
     */
    public static function billOutstandingAmount($profileId)
    {
        $data = array();
        $data = self::addCredentials($data);
        if (!empty($data['error'])) {
            return array('error' => '1', $data['message']);
        }

        $action = self::getEndpointUrl();
        $data['METHOD'] = self::BILL_OUTSTANDING_AMOUNT;
        $data['PROFILEID'] = $profileId;

        $response = self::makeRequest('POST', $action, $data);

        if (!empty($response['L_ERRORCODE0'])) {
            return array('error' => '1', 'message' => $response['L_LONGMESSAGE0']);
        }

        return $response;
    }

    /**
     * @return string
     */
    public static function getEndpointUrl()
    {
        $paypalConfigMapper = Paypal_Models_Mapper_PaypalConfigMapper::getInstance();
        $paypalConfig = $paypalConfigMapper->getConfigParams();
        $action = self::LIVE_API_GATEWAY_URL;
        if (!empty($paypalConfig['useSandbox'])) {
            $action = self::SANDBOX_API_GATEWAY_URL;
        }

        return $action;
    }


    /**
     * @param array $data
     * @return array
     */
    public static function addCredentials($data)
    {

        $paypalConfigMapper = Paypal_Models_Mapper_PaypalConfigMapper::getInstance();
        $paypalConfig = $paypalConfigMapper->getConfigParams();

        if (empty($paypalConfig)) {
            return array('error' => '1', 'Empty config');
        }

        $data['USER'] = $paypalConfig['apiUser'];
        $data['PWD'] = $paypalConfig['apiPassword'];
        $data['SIGNATURE'] = $paypalConfig['apiSignature'];
        $data['VERSION'] = self::API_VERSION;

        return $data;
    }

    /**
     * Request remote service API
     *
     * @param string $method 'POST', 'GET', 'PUT', 'DELETE'
     * @param string $action action url
     * @param array $data data to send
     * @return mixed
     */
    public static function makeRequest($method, $action, $data)
    {
        if (strtoupper($method) == 'GET' || strtoupper($method) == 'DELETE') {
            foreach ($data as $k => $v) {
                if (is_array($v)) {
                    $v = serialize($v);
                }
                $action .= '/' . $k . '/' . $v;
            }
        }
        $curl = curl_init();
        curl_setopt($curl, CURLOPT_URL, $action);
        curl_setopt($curl, CURLOPT_HEADER, false);
        curl_setopt($curl, CURLOPT_VERBOSE, 1);
        curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($curl, CURLOPT_TIMEOUT, 20);
        $nvpRequest = '';
        if (!empty($data)) {
            if (strtoupper($method) == 'POST') {
                $nvpRequest = http_build_query($data);
            }
            if (strtoupper($method) == 'PUT') {
                $nvpRequest = json_encode($data);
            }
        }
        if (strtoupper($method) == 'POST') {
            curl_setopt($curl, CURLOPT_POST, 1);
            curl_setopt($curl, CURLOPT_POSTFIELDS, $nvpRequest);
        } elseif (strtoupper($method) == 'PUT') {
            curl_setopt($curl, CURLOPT_HTTPHEADER, array(
                    'Content-Type: application/json',
                    'Content-Length: ' . strlen($nvpRequest)
                )
            );
            curl_setopt($curl, CURLOPT_CUSTOMREQUEST, strtoupper($method));
            curl_setopt($curl, CURLOPT_POSTFIELDS, $nvpRequest);
        } elseif (strtoupper($method) == 'DELETE') {
            curl_setopt($curl, CURLOPT_CUSTOMREQUEST, strtoupper($method));
        }
        $response = curl_exec($curl);
        curl_close($curl);
        $responseResult = array();

        parse_str($response, $responseResult);

        return $responseResult;
    }

    /**
     * Update information about profile in the local database
     *
     * @param int $profileId profile id
     * @param int $userId user id
     * @param int $cartId cart id
     * @return array
     * @throws Exceptions_SeotoasterException
     */
    public static function updateLocalProfileInfo($profileId, $userId, $cartId = 0)
    {
        $profileInfo = Tools_RecurringPaymentPaypal::getRecurringPaymentsProfileDetails($profileId);
        if (!empty($profileInfo['error']) && $profileInfo['error'] === '1') {
            return $profileInfo;
        }

        $status = $profileInfo['STATUS'];
        $autoBillOutAmt = $profileInfo['AUTOBILLOUTAMT'];
        $desc = $profileInfo['DESC'];
        $maxFailedPayments = $profileInfo['MAXFAILEDPAYMENTS'];
        $profileStartDate = $profileInfo['PROFILESTARTDATE'];
        $nextBillingDate = '';
        if (!empty($profileInfo['NEXTBILLINGDATE'])) {
            $nextBillingDate =  $profileInfo['NEXTBILLINGDATE'];
        }
        $numCyclesCompleted = $profileInfo['NUMCYCLESCOMPLETED'];
        $numCyclesRemaining = $profileInfo['NUMCYCLESREMAINING'];
        $outstandingBalance = $profileInfo['OUTSTANDINGBALANCE'];
        $failedPaymentCount = $profileInfo['FAILEDPAYMENTCOUNT'];
        $lastPaymentDate = $profileInfo['LASTPAYMENTDATE'];
        $lastPaymentAmt = $profileInfo['LASTPAYMENTAMT'];
        $trialAmtPaid = $profileInfo['TRIALAMTPAID'];
        $regularAmtPaid = $profileInfo['REGULARAMTPAID'];
        $aggregateAmt = $profileInfo['AGGREGATEAMT'];
        $aggregateOptionalAmt = $profileInfo['AGGREGATEOPTIONALAMT'];
        $finalPaymentDueDate = $profileInfo['FINALPAYMENTDUEDATE'];
        $timestap = $profileInfo['TIMESTAMP'];
        $correlationId = $profileInfo['CORRELATIONID'];
        $billingPeriod = $profileInfo['BILLINGPERIOD'];
        $billingFrequency = $profileInfo['BILLINGFREQUENCY'];
        $totalBillingCycles = $profileInfo['TOTALBILLINGCYCLES'];
        $currencyCode = $profileInfo['CURRENCYCODE'];
        $amt = $profileInfo['AMT'];
        $shippingAmt = $profileInfo['SHIPPINGAMT'];
        $taxAmt = $profileInfo['TAXAMT'];
        $regularBillingPeriod = $profileInfo['REGULARBILLINGPERIOD'];
        $regularBillingFrequency = $profileInfo['REGULARBILLINGFREQUENCY'];
        $regularTotalBillingCycles = $profileInfo['REGULARTOTALBILLINGCYCLES'];
        $regularCurrencyCode = $profileInfo['REGULARCURRENCYCODE'];
        $regularAmt = $profileInfo['REGULARAMT'];
        $regularShippingAmt = $profileInfo['REGULARSHIPPINGAMT'];
        $regularTaxAmt = $profileInfo['REGULARTAXAMT'];
        $acct = $profileInfo['ACCT'];
        $creditCardType = $profileInfo['CREDITCARDTYPE'];
        $expDate = $profileInfo['EXPDATE'];
        if (!empty($profileInfo['EMAIL'])) {
            $email = $profileInfo['EMAIL'];
        }

        $street = $profileInfo['STREET'];
        $city = $profileInfo['CITY'];
        $state = $profileInfo['STATE'];
        $zip = $profileInfo['ZIP'];
        $countryCode = $profileInfo['COUNTRYCODE'];
        $country = $profileInfo['COUNTRY'];
        $addressOwner = $profileInfo['ADDRESSOWNER'];
        $addressStatus = $profileInfo['ADDRESSSTATUS'];
        $payerStatus = $profileInfo['PAYERSTATUS'];

        $paypalRecurringPaymentsMapper = Paypal_Models_Mapper_PaypalRecurringPaymentsMapper::getInstance();
        $paypalRecurringPaymentsModel = $paypalRecurringPaymentsMapper->findByProfileId($profileId);
        if (!$paypalRecurringPaymentsModel instanceof Paypal_Models_Models_PaypalRecurringPaymentsModel) {
            $paypalRecurringPaymentsModel = new Paypal_Models_Models_PaypalRecurringPaymentsModel();
            $paypalRecurringPaymentsModel->setProfileId($profileId);
        }

        if (empty($email)) {
            $email = 'Not provided';
        }

        $paypalRecurringPaymentsModel->setUserId($userId);
        $paypalRecurringPaymentsModel->setCartId($cartId);
        $paypalRecurringPaymentsModel->setStatus($status);
        $paypalRecurringPaymentsModel->setAutoBillOutAmt($autoBillOutAmt);
        $paypalRecurringPaymentsModel->setDesc($desc);
        $paypalRecurringPaymentsModel->setMaxFailedPayments($maxFailedPayments);
        $paypalRecurringPaymentsModel->setProfileStartDate($profileStartDate);
        $paypalRecurringPaymentsModel->setNextBillingDate($nextBillingDate);
        $paypalRecurringPaymentsModel->setNumCyclesCompleted($numCyclesCompleted);
        $paypalRecurringPaymentsModel->setNumCyclesRemaining($numCyclesRemaining);
        $paypalRecurringPaymentsModel->setOutstandingBalance($outstandingBalance);
        $paypalRecurringPaymentsModel->setFailedPaymentCount($failedPaymentCount);
        $paypalRecurringPaymentsModel->setLastPaymentDate($lastPaymentDate);
        $paypalRecurringPaymentsModel->setLastPaymentAmt($lastPaymentAmt);
        $paypalRecurringPaymentsModel->setTrialAmtPaid($trialAmtPaid);
        $paypalRecurringPaymentsModel->setRegularAmtPaid($regularAmtPaid);
        $paypalRecurringPaymentsModel->setAggregateAmt($aggregateAmt);
        $paypalRecurringPaymentsModel->setAggregateOptionalAmt($aggregateOptionalAmt);
        $paypalRecurringPaymentsModel->setFinalPaymentDueDate($finalPaymentDueDate);
        $paypalRecurringPaymentsModel->setTimestamp($timestap);
        $paypalRecurringPaymentsModel->setCorrelationId($correlationId);
        $paypalRecurringPaymentsModel->setBillingPeriod($billingPeriod);
        $paypalRecurringPaymentsModel->setBillingFrequency($billingFrequency);
        $paypalRecurringPaymentsModel->setTotalBillingCycles($totalBillingCycles);
        $paypalRecurringPaymentsModel->setCurrencyCode($currencyCode);
        $paypalRecurringPaymentsModel->setAmt($amt);
        $paypalRecurringPaymentsModel->setShippingAmt($shippingAmt);
        $paypalRecurringPaymentsModel->setTaxAmt($taxAmt);
        $paypalRecurringPaymentsModel->setRegularBillingPeriod($regularBillingPeriod);
        $paypalRecurringPaymentsModel->setRegularBillingFrequency($regularBillingFrequency);
        $paypalRecurringPaymentsModel->setRegularTotalBillingCycles($regularTotalBillingCycles);
        $paypalRecurringPaymentsModel->setRegularCurrencyCode($regularCurrencyCode);
        $paypalRecurringPaymentsModel->setRegularAmt($regularAmt);
        $paypalRecurringPaymentsModel->setRegularShippingAmt($regularShippingAmt);
        $paypalRecurringPaymentsModel->setRegularTaxAmt($regularTaxAmt);
        $paypalRecurringPaymentsModel->setAcct($acct);
        $paypalRecurringPaymentsModel->setCreditCardType($creditCardType);
        $paypalRecurringPaymentsModel->setExpDate($expDate);
        $paypalRecurringPaymentsModel->setEmail($email);
        $paypalRecurringPaymentsModel->setStreet($street);
        $paypalRecurringPaymentsModel->setCity($city);
        $paypalRecurringPaymentsModel->setState($state);
        $paypalRecurringPaymentsModel->setZip($zip);
        $paypalRecurringPaymentsModel->setCountryCode($countryCode);
        $paypalRecurringPaymentsModel->setCountry($country);
        $paypalRecurringPaymentsModel->setAddressOwner($addressOwner);
        $paypalRecurringPaymentsModel->setAddressStatus($addressStatus);
        $paypalRecurringPaymentsModel->setPayerStatus($payerStatus);

        $paypalRecurringPaymentsModel = $paypalRecurringPaymentsMapper->save($paypalRecurringPaymentsModel);

        return array('error' => '0', 'paypalRecurringPaymentsModel' => $paypalRecurringPaymentsModel);
    }
}