<?php
require __DIR__ . '/system/library/minishlink/vendor/autoload.php';

use Minishlink\WebPush\WebPush;
use Minishlink\WebPush\VAPID;

class Notifier extends Tools_Plugins_Abstract
{

    /**
     * Firebase not registered error constant
     */
    const FIREBASE_NOT_REGISTERED = 'Error=NotRegistered';

    /**
     * Firebase success notif response const
     */
    const FIREBASE_SUCCESS = 'id=';


    const NOTIFIER_SECURE_TOKEN = 'NotifierToken';

    const CRON_SCRIPT_PROCESS_PENDING_NEWS_PATH = 'plugins/notifier/system/cronscript/processPendingNews.php';

    /**
     * Toaster layout
     *
     * @var
     */
    private $_layout = null;

    protected $_notifierConfigMapper = null;

    public function __construct($options, $seotoasterData)
    {
        parent::__construct($options, $seotoasterData);
        $this->_layout = new Zend_Layout();
        $this->_layout->setLayoutPath(Zend_Layout::getMvcInstance()->getLayoutPath());
        $this->_notifierConfigMapper = Notifier_Models_Mappers_NotifierConfigMapper::getInstance();

        if (($scriptPaths = Zend_Layout::getMvcInstance()->getView()->getScriptPaths()) !== false) {
            $this->_view->setScriptPath($scriptPaths);
        }
        $this->_view->addScriptPath(__DIR__ . '/system/views/');
    }

    public function configAction()
    {
        if (Tools_Security_Acl::isAllowed(Tools_Security_Acl::RESOURCE_PLUGINS)) {
            $notifierConfig = $this->_notifierConfigMapper->getConfigParams();
            $notifierConfig['subscriptionsQty'] = count(Notifier_Models_Mappers_NotifierSubscriptionsMapper::getInstance()->fetchAll());
            if ($this->_request->isPost()) {
                $senderId = filter_var($this->_request->getParam('senderId'), FILTER_SANITIZE_NUMBER_INT);
                $serverKey = filter_var($this->_request->getParam('serverKey', FILTER_SANITIZE_STRING));
                $useCustomPopup = filter_var($this->_request->getParam('useCustomPopup', FILTER_VALIDATE_BOOLEAN));
                $cronNotifications = filter_var($this->_request->getParam('cronNotifications', FILTER_VALIDATE_BOOLEAN));
                $headerText = filter_var($this->_request->getParam('headerText', FILTER_SANITIZE_STRING));
                $allowLabel = filter_var($this->_request->getParam('allowLabel', FILTER_SANITIZE_STRING));
                $blockLabel = filter_var($this->_request->getParam('blockLabel', FILTER_SANITIZE_STRING));
                $messageText = filter_var($this->_request->getParam('messageText', FILTER_SANITIZE_STRING));
                $headerColor = filter_var($this->_request->getParam('headerColor', FILTER_SANITIZE_STRING));
                $allowBgColor = filter_var($this->_request->getParam('allowBgColor', FILTER_SANITIZE_STRING));
                $allowHvColor = filter_var($this->_request->getParam('allowHvColor', FILTER_SANITIZE_STRING));
                $allowFtColor = filter_var($this->_request->getParam('allowFtColor', FILTER_SANITIZE_STRING));
                $blockBgColor = filter_var($this->_request->getParam('blockBgColor', FILTER_SANITIZE_STRING));
                $blockHvColor = filter_var($this->_request->getParam('blockHvColor', FILTER_SANITIZE_STRING));
                $blockFtColor = filter_var($this->_request->getParam('blockFtColor', FILTER_SANITIZE_STRING));
                $backgroundColor = filter_var($this->_request->getParam('backgroundColor', FILTER_SANITIZE_STRING));
                $fontColor = filter_var($this->_request->getParam('fontColor', FILTER_SANITIZE_STRING));
                $timeInterval = filter_var($this->_request->getParam('timeInterval', FILTER_SANITIZE_NUMBER_INT));
                $logoSize = filter_var($this->_request->getParam('logoSize', FILTER_SANITIZE_STRING));
                $fontStyle = filter_var($this->_request->getParam('fontStyle', FILTER_SANITIZE_STRING));
                $fontSize = filter_var($this->_request->getParam('fontSize', FILTER_SANITIZE_STRING));
                $headerSize = filter_var($this->_request->getParam('headerSize', FILTER_SANITIZE_STRING));

                $secureToken = $this->_request->getParam(Tools_System_Tools::CSRF_SECURE_TOKEN, false);
                $tokenValid = Tools_System_Tools::validateToken($secureToken, self::NOTIFIER_SECURE_TOKEN);
                if (!$tokenValid) {
                    $this->_responseHelper->fail('');
                }

                if (empty($serverKey)) {
                    if (isset($notifierConfig['serverKey'])) {
                        $serverKey = $notifierConfig['serverKey'];
                    }
                }

                if (empty($senderId) || empty($serverKey)) {
                    $this->_responseHelper->fail($this->_translator->translate('Notifier was not configured!'));
                }

                if ($useCustomPopup == "true" && empty($messageText)) {
                    $this->_responseHelper->fail($this->_translator->translate('Message cannot be empty!'));
                }

                if (!file_exists('manifest.json')) {
                    $this->_responseHelper->fail($this->_translator->translate('Notifier could not locate manifest.json'));
                }
                // Update Manifest
                $manifestArray = json_decode(Tools_Filesystem_Tools::getFile('manifest.json'), true);
                $manifestArray['gcm_sender_id'] = $senderId;
                try {
                    Tools_Filesystem_Tools::saveFile('manifest.json', json_encode($manifestArray, JSON_UNESCAPED_SLASHES));
                } catch (Exception $e) {
                    $this->_responseHelper->fail($this->_translator->translate('Notifier could not re-write manifest.json'));
                }
                $params = [
                    'senderId' => $senderId,
                    'serverKey' => $serverKey
                ];
                // Check if VAPID keys exist
                if (!$notifierConfig['publicKey'] && !$notifierConfig['privateKey']) {
                    $params = array_merge(VAPID::createVapidKeys(), $params);
                }
                $params['useCustomPopup'] = $useCustomPopup;
                $params['cronNotifications'] = $cronNotifications;
                if ($useCustomPopup === "true") {
                    $params['headerText'] = $headerText ? $headerText : '';
                    $params['allowLabel'] = $allowLabel ? $allowLabel : 'Allow';
                    $params['blockLabel'] = $blockLabel ? $blockLabel : 'Block';
                    $params['messageText'] = $messageText ? $messageText : 'Do you allow push-notifications?';
                    $params['backgroundColor'] = !empty($backgroundColor) ? $backgroundColor : '#ffffff';
                    $params['headerColor'] = !empty($headerColor) ? $headerColor : '#ffffff';
                    $params['fontColor'] = !empty($fontColor) ? $fontColor : '#0e0e0e';
                    $params['allowBgColor'] = !empty($allowBgColor) ? $allowBgColor : '';
                    $params['allowHvColor'] = !empty($allowHvColor) ? $allowHvColor : '';
                    $params['allowFtColor'] = !empty($allowFtColor) ? $allowFtColor : '';
                    $params['blockBgColor'] = !empty($blockBgColor) ? $blockBgColor : '';
                    $params['blockHvColor'] = !empty($blockHvColor) ? $blockHvColor : '';
                    $params['blockFtColor'] = !empty($blockFtColor) ? $blockFtColor : '';
                    $params['timeInterval'] = $timeInterval ? $timeInterval : 10;
                    $params['logoSize'] = $logoSize ? $logoSize : 'medium';
                    $params['fontStyle'] = $fontStyle ? $fontStyle : '';
                    $params['fontSize'] = $fontSize ? $fontSize : '14';
                    $params['headerSize'] = $headerSize ? $headerSize : '14';
                }

                $this->_notifierConfigMapper->save($params);
                $this->_responseHelper->success('');
            } else {
                $fontSizes = [];
                for ($i = 8; $i <= 24; $i++) {
                    $fontSizes[] = $i;
                }
                $this->_view->settings = $notifierConfig;
                $this->_view->fontSizes = $fontSizes;
                $this->_view->translator = $this->_translator;
                $this->_view->cronPath = Zend_Registry::get('website')['path'] . self::CRON_SCRIPT_PROCESS_PENDING_NEWS_PATH;
                $this->_layout->content = $this->_view->render('notifierConfig.phtml');
                echo $this->_layout->render();
            }
        }
    }

    public function _makeOptionSubscribeButton()
    {
        $notifierConfig = $this->_notifierConfigMapper->getConfigParams();
        $this->_view->publicKey = !empty($notifierConfig['publicKey']) ? $notifierConfig['publicKey'] : '';
        $this->_view->buttonText = !empty($this->_options[1]) ? filter_var($this->_options[1], FILTER_SANITIZE_STRING) : $this->_translator->translate('Subscribe');
        $this->_view->settings = $notifierConfig;
        return $this->_view->render('subscribeButton.phtml');
    }

    public function _makeOptionSubscribe()
    {
        $notifierConfig = $this->_notifierConfigMapper->getConfigParams();
        $this->_view->publicKey = !empty($notifierConfig['publicKey']) ? $notifierConfig['publicKey'] : '';
        $this->_view->settings = $notifierConfig;
        if (isset($notifierConfig['logoSize'])) {
            switch ($notifierConfig['logoSize']) {
                case 'small':
                    $this->_view->settings['logoPath'] = '/plugins/widcard/system/userdata/icons/apple-icon-60x60.png';
                    $this->_view->settings['logoSize'] = '60';
                    break;
                case 'medium':
                    $this->_view->settings['logoPath'] = '/plugins/widcard/system/userdata/icons/app-icon-96x96.png';
                    $this->_view->settings['logoSize'] = '96';
                    break;
                case 'large':
                    $this->_view->settings['logoPath'] = '/plugins/widcard/system/userdata/icons/apple-icon-120x120.png';
                    $this->_view->settings['logoSize'] = '120';
                    break;
                default:
                    $this->_view->settings['logoPath'] = '/plugins/widcard/system/userdata/icons/app-icon-96x96.png';
                    $this->_view->settings['logoSize'] = '96';
            }
        } else {
            $this->_view->settings['logoPath'] = '/plugins/widcard/system/userdata/icons/app-icon-96x96.png';
            $this->_view->settings['logoSize'] = '96';
        }
        return $this->_view->render('subscribe.phtml');
    }

    /**
     * Notifier news open endpoint
     * Send notifications, remove expired subscriptions
     * @return bool
     */
    public function notifyNewsAction()
    {
        $notifierConfig = $this->_notifierConfigMapper->getConfigParams();

        $auth = [
            'VAPID' => [
                'subject' => 'mailto:support@seosamba.com',
                'publicKey' => $notifierConfig['publicKey'],
                'privateKey' => $notifierConfig['privateKey'],
            ],
        ];

        $newsNotifications = Notifier_Models_Mappers_NotifierNewsMapper::getInstance()->fetchWaitingWithNews();
        $webPush = new WebPush($auth);

        foreach ($newsNotifications as $notification) {
            $meta = json_decode($notification['metaData']);

            // Create subscription object
            $subscription = \Minishlink\WebPush\Subscription::create([
                'endpoint' => $notification['endpoint'],
                'publicKey' => $notification['p256dh'],
                'authToken' => $notification['auth'],
                'contentEncoding' => 'aesgcm', // or 'aes128gcm', adjust if needed
            ]);

            // Send notification
            $res = $webPush->sendOneNotification(
                $subscription,
                json_encode([
                    'title' => $meta->title,
                    'content' => $meta->teaserText,
                    'openUrl' => $meta->url,
                    'image' => '/previews/' . $meta->image,
                ])
            );

            // Adapted expired check
            $expired = false;
            if (!$res->isSuccess()) {
                $reason = $res->getReason();
                if (preg_match('/gone|expire|unsubscribe/i', $reason)) {
                    $expired = true;
                }
            }

            if ($expired) {
                Notifier_Models_Mappers_NotifierSubscriptionsMapper::getInstance()->delete($notification['token']);
            } else {
                Notifier_Models_Mappers_NotifierNewsMapper::getInstance()->updateStatus($notification['id']);
            }
        }

        return true;

    }
}
