import {createNamespacedHelpers} from 'vuex';
import {mapGetters as rootMap} from 'vuex';
import moment from 'moment';
import localeMapping from '../../../localizationLanguages';

const {mapActions, mapGetters, mapMutations} = createNamespacedHelpers('aiChat')
export default {
    data() {
        return {
            prompt: '',
            message: '',
            loading: false,
            showChat: false,
            isAnswering: false,
            websiteUrl: $('#website_url').val(),
            localeMapping: localeMapping,
            locale: $('#dashboard-system-language').val(),
            timezoneAbbr:'',
            timezoneOffset:''
        }
    },
    computed: {
        ...rootMap({
            rootState: 'getRootState'
        }),
        ...mapGetters({
            state: 'getState'
        }),
        creditBalance(){
            return Intl.NumberFormat('en-US', {
                notation: "compact",
                maximumFractionDigits: 1
            }).format(this.state.creditBalance);
        },
        styleOptions(){
                return {button: {}, header: {}}
        }

    },

    methods: {
        ...mapActions([
            'getCreditBalance',
            'getHistoryAction',
            'postHistoryAction'
        ]),
        async showChatAction(){
            const resultHistory = await this.getHistoryAction();
            const resultCreditBalance = await this.getCreditBalance();

            if (parseInt(resultHistory.error) === 1) {
                showMessage(resultHistory.message, true, 3000);
                return false;
            }

            if (parseInt(resultCreditBalance.error) === 1) {
                showMessage(resultCreditBalance.message, true, 3000);
                return false;
            }

            this.timezoneAbbr = resultCreditBalance.timezoneAbbr;
            this.timezoneOffset = resultCreditBalance.timezoneOffset;

            this.showChat = true;
            this.$nextTick(() => this.$refs.chatBlock.scrollTop = this.$refs.chatBlock.lastElementChild.offsetTop);
        },
        copyChat(){
            let message = '';
            if(this.state.messages.length){
                this.state.messages.forEach(item => {
                    if(item.role === 'assistant'){
                        message += `AI Assistant: ${item.content} - ${this.time(item.timestamp)} \n\n`;
                    }
                    if(item.role === 'user'){
                        message += `You: ${item.content} - ${this.time(item.timestamp)} \n\n`;
                    }
                });
                this.copyContent(message);
            }
        },
        copyContent(message){
            let self = this;

            this.$copyText(message).then(function (e) {
                showMessage(self.$t('message.copiedToClipboard'), false, 2000)
            }, function (e) {
                showMessage(self.$t('message.cantCopy'), true, 3000);
            });
        },
        time(time){
            let date = moment(time);
            if(!time){
                date = moment(moment().utc());
                if (parseInt(this.timezoneOffset) > 0) {
                    date.add(Math.abs(parseInt(this.timezoneOffset)), 'hours');
                } else {
                    date.subtract(Math.abs(parseInt(this.timezoneOffset)), 'hours');
                }
             }

            let sameDayFormat = this.$t('message.todayAt');
            let lastDayFormat = this.$t('message.yesterdayAt');
            let at = this.$t('message.at');

            return date.calendar(null, {
                sameDay: '['+sameDayFormat+']'+ ' hh:mm a',
                lastDay: '['+lastDayFormat+']'+ ' hh:mm a',
                lastWeek: 'DD MMM YYYY '+ '['+at+']'+ ' hh:mm a',
                sameElse: 'DD MMM YYYY '+ '['+at+']'+ ' hh:mm a'
            }) + ' ' + this.timezoneAbbr;
        },
        async clear(){
            this.loading = true;
            await this.postHistoryAction({clear: true});
            await this.getHistoryAction();
            this.loading = false;
        },
        async exportHistory(){
            const response = await this.postHistoryAction({export: true});
            if(!response){
                showMessage(this.$t('message.somethingWentWrong'), true, 3000);
                return;
            }
            const link = $("<a>").attr({href: response.link, target: '_blank', download: 'AI-conversation-export.csv'});
            $('body').append(link);
            link.get(0).click();
            link.remove();
        },
       async  loadHistory(){
            this.loading = true;
            await this.getHistoryAction({history: true});
            this.loading = false;
            this.$nextTick(() => this.$refs.chatBlock.scrollTop = this.$refs.chatBlock.lastElementChild.offsetTop);

        },
        async sendMessage(){
            if(!this.prompt || this.prompt.length > 1000){
                return;
            }
            this.loading = true;

            const res = await fetch(`${this.websiteUrl}api/leadsopenai/openaieventstream`, {
                method: "POST",
                headers: { "Content-Type": "application/x-www-form-urlencoded" },
                body: `prompt=${encodeURIComponent(this.prompt)}`
            });

            const { streamId } = await res.json();

            const eventSource = new EventSource(`${this.websiteUrl}api/leadsopenai/openaieventstream?id=${streamId}`);
            this.$store.commit('aiChat/setState', {messages: [...this.state.messages, {role: 'user', content: this.prompt}]})
            this.prompt = '';
            eventSource.onmessage =  (e) => {
                if (e.data == "[DONE]") {
                    let message = this.message.replace(
                        /\(\[([^\]]+)\]\((https?:\/\/[^)]+)\)\)/g,
                        `<a href="$2" target="_blank" rel="noopener noreferrer">$1</a>`
                    );

                    // 2) Remove utm_source=openai
                    message = message.replace(/[?&]utm_source=openai/g, '');

                    // --- Convert string to DOM ---
                    const parser = new DOMParser();
                    const doc = parser.parseFromString(message, 'text/html');

                    // --- Remove ONLY the second link when its text is "help.seosamba.com" ---
                    doc.querySelectorAll('*').forEach(parent => {
                        const links = parent.querySelectorAll(':scope > a'); // only direct children
                        if (links.length === 2) {
                            const second = links[1];
                            if (second.textContent.trim() === 'help.seosamba.com') {
                                second.remove();
                            }
                        }
                    });

                    // --- Back to string ---
                    message = doc.body.innerHTML;

                    this.$store.commit('aiChat/setState', {messages: [...this.state.messages, {role: 'assistant', content: message}]})
                    this.message = '';
                    this.getCreditBalance();
                    eventSource.close();
                    this.loading = false;
                    this.isAnswering = false;
                    this.$refs.chatBlock.scrollTop = this.$refs.chatBlock.lastElementChild.offsetTop
                } else {
                    this.isAnswering = true;
                    //this.message += ' '+ e.data;
                    const response = JSON.parse(e.data);
                    if(response.error){
                        showMessage(response.message, response.error);
                        eventSource.close();
                        this.isAnswering = false;
                        return;
                    }
                    //let content = response.toString() + ' ;';
                    if(response?.choices?.length) {
                        let content = response.choices[0].delta.content
                        if (content) {
                            this.message += content;
                        }
                    }
                    if(response.text){
                        this.message += response.text;
                    }
                    this.$refs.chatBlock.scrollTop = this.$refs.chatBlock.lastElementChild.offsetTop;
                }
            };
            eventSource.onerror =  (e) => {
                console.log(e);
                this.loading = false;
                this.isAnswering = false;
                showMessage(this.$t('message.somethingWentWrong'), true, 3000);
                eventSource.close();
                this.getCreditBalance();
            };
        }
    },
    created(){
        if (typeof this.localeMapping[this.locale] !== 'undefined') {
            this.$i18n.locale = this.localeMapping[this.locale];
        }
    }
}