<?php
/**
 * News list widget {$newslist:template_name[:list_name[:per_page[:asc|desc[:pgr[:tags:tag1,tag2, ..., tagN]]]]}
 *
 * template_name -required parameter, represents news list template that will be loaded for rendering
 * list_name - optional, unique news list identifier
 * per_page - optional parameter, how many news items should be displayed in list
 * asc|desc - order direction. DESC used by default
 * pgr - use pager or not
 * tags:tag1,tag2,...,tagN - List of tags to filter a news list
 *
 * User: Eugene I. Nezhuta <eugene@seosamba.com>
 * Date: 10/25/13
 * Time: 5:43 PM
 */

class Widgets_Newslist_Newslist extends Widgets_Abstract {

    /**
     * News widget name in the system
     *
     */
    const NEWS_WIDGET_NAME  = 'news';

    /**
     * List order direction (LOD) ascendant
     *
     */
    const LOD_ASC           = 'asc';

    /**
     * List order direction (LOD) descendant
     *
     */
    const LOD_DESC          = 'desc';

    /**
     * List option tags
     *
     */
    const OPT_TAGS          = 'tags';

    /**
     * List options pager
     *
     */
    const OPT_PAGER         = 'pgr';

    /**
     * Default news list name
     *
     */
    const DEFAULT_LIST_NAME = 'newslist';

    /**
     * Date param option
     *
     */
    const OPT_DATE         = 'maxdate-';


    const OPT_SHOW_MORE         = 'showmore';

    /**
     * Exclude events flag
     */
    const NEWS_EXCLUDE_EVENTS = 'exclude-events';

    /**
     * @var bool
     */
    protected $_cleanListOnly = false;

    /**
     * @var array
     */
    private $_news        = array();

    /**
     * News list order direction. Descending by default.
     *
     * @var string
     */
    protected $_orderDirection = self::LOD_DESC;

    /**
     * How many news items should be in news list. Leave null to get everything.
     *
     * @var mixed integer|null
     */
    protected $_limit         = null;

    /**
     * Display news that have specific tags
     *
     * @var array
     */
    protected $_tags          = array();

    /**
     * News list name. Used for the filtering by tags and pagination
     *
     * @var string
     */
    protected $_listName      = self::DEFAULT_LIST_NAME;

    /**
     * News filter flag. Used for the filtering by tags and pagination in all lists
     *
     * @var string
     */
    protected $_listAll      = true;

    /**
     * Use pagination or not in news list
     *
     * @var bool
     */
    protected $_pagination    = false;

    /**
     * @var bool
     */
    protected $_showmore    = false;

    /**
     * @var string
     */
    protected $_orderByCol = 'p.publish_at';

    protected $_allowedOrderParams = array(
        'publish-at-sort',
        'event-sort',
        'created-at-sort'
    );

    protected $_allowedOrderColumnsNames = array(
        'publish-at-sort' => 'p.publish_at',
        'event-sort' => 'n.event_date',
        'created-at-sort' => 'n.created_at'
    );

    /**
     * Do not cache news list widget by default
     *
     * @var bool
     */
    protected $_cacheable       = false;

    /**
     * If true widget will also put a record to the error log file
     *
     * @var bool
     */
    private $_debugMode         = false;

    /**
     * News mapper
     *
     * @var Newslog_Models_Mapper_NewsMapper
     */
    protected $_mapper          = null;

    protected $_excludeEvents = false;

    /**
     * Website helper with a lot of useful data
     *
     * @var null
     */
    protected $_websiteHelper   = null;

    protected function _init() {
        $this->_view             = new Zend_View(array('scriptPath' => __DIR__ . '/views'));
        $this->_view->setHelperPath(APPLICATION_PATH . '/views/helpers/');
        $this->_view->addHelperPath('ZendX/JQuery/View/Helper/', 'ZendX_JQuery_View_Helper');

        $this->_websiteHelper    = Zend_Controller_Action_HelperBroker::getStaticHelper('website');
        $this->_view->websiteUrl = $this->_websiteHelper->getUrl();
        $this->_mapper           = Newslog_Models_Mapper_NewsMapper::getInstance();
        $this->_debugMode        = Tools_System_Tools::debugMode();
    }

    protected function _load() {
        // loading news list template
        $templateName = array_shift($this->_options);
        $this->_config = Zend_Controller_Action_HelperBroker::getStaticHelper('config');

        // if developerMode = 1, parsing template directly from files
        if ((bool) $this->_config->getConfig('enableDeveloperMode')) {
            $websiteHelper = Zend_Controller_Action_HelperBroker::getExistingHelper('website');
            $themeData = Zend_Registry::get('theme');
            $extConfig = Zend_Registry::get('extConfig');

            $websitePath  = $websiteHelper->getPath();
            $themePath    = $themeData['path'];
            $currentTheme = $extConfig['currentTheme'];
            $templatePath = $websitePath.$themePath.$currentTheme.DIRECTORY_SEPARATOR.$templateName.'.html';
            if (file_exists($templatePath)) {
                $templateContent = Tools_Filesystem_Tools::getFile($templatePath);
            } else {
                $templateContent = '<span style="color: red;">News list template "<em>' . $templateName . '</em>" not found</span>';
            }
        }else{
            $template = Application_Model_Mappers_TemplateMapper::getInstance()->find($templateName);
            if($template === null) {
                $templateContent = '<span style="color: red;">News list template "<em>' . $templateName . '</em>" not found</span>';
            }else {
                $templateContent = $template->getContent();
            }
        }

        // parse options passed to the widget and init proper vars
        $this->_parseOptions();

        $publishAtSortFlag = false;
        if ($this->_orderByCol === 'p.publish_at') {
            $publishAtSortFlag = true;
        }

        // Check the request for pagination or tags filtering
        $request = Zend_Controller_Front::getInstance()->getRequest();
        if(($request->has('list') && ($this->_listName == $request->getParam('list', self::DEFAULT_LIST_NAME))) || ($request->has('listall') && ($this->_listAll == $request->getParam('listall')))) {
            $this->_tags = $request->getParam('tags', array());
            if(!is_array($this->_tags)) {
                $this->_tags = explode(',', $this->_tags);
            }
        }

        $order[] = $this->_orderByCol . ' ' . $this->_orderDirection;
        if ($publishAtSortFlag) {
            $order[] = 'n.created_at' . ' ' . $this->_orderDirection;
        }

        if(in_array(self::OPT_SHOW_MORE, $this->_options)) {
            $this->_showmore = true;
        }

        $where = 'n.published = 1';
        if ($this->_excludeEvents === true) {
            $where .= ' AND '. $this->_mapper->getDbtable()->getAdapter()->quoteInto('n.event = ?', '0');
        }

        $excludeTags = array();
        if(in_array('excludetags', $this->_options)) {
            $tagsKey = array_search('excludetags', $this->_options);
            $excludeTags = array_filter(explode(',', $this->_options[$tagsKey + 1]));
        }

        $excludedNewsIds = array();
        if(!empty($excludeTags)) {
            $this->_tags = array();

            $excludedNewsIds = Newslog_Models_Mapper_TagMapper::getInstance()->findNewsIdsByTagsNames($excludeTags);
        }

        // loading news
        if ($this->_pagination && !$this->_showmore) {

            $dateParam = false;
            $dateOpt = preg_grep("/" . self::OPT_DATE . "/i", $this->_options);
            $serchedStrDate = array_pop($dateOpt);
            if (!empty($serchedStrDate)) {
                $dateParam = str_replace(self::OPT_DATE, '', $serchedStrDate);
            }
            $date = DateTime::createFromFormat('Y-m-d', $dateParam);
            if (!empty($date) && date('Y-m-d') >= $dateParam) {
                $where .= " AND n.created_at >= '" . $dateParam . " 00:00:00'";
            }

            $news = Zend_Paginator::factory($this->_mapper->fetchAll($where, $order, null, 0, $this->_tags, null, false, $excludedNewsIds));
            $news->setDefaultItemCountPerPage($this->_limit);
            $news->setCurrentPageNumber(1);
            $news->setDefaultPageRange(3);
            if ($this->_listName == $request->getParam('list',
                    self::DEFAULT_LIST_NAME) || ($this->_listAll == $request->getParam('listall'))
            ) {
                $news->setCurrentPageNumber(filter_var($request->getParam('pg', 1), FILTER_SANITIZE_NUMBER_INT));
            }

            //initialize view to render pagination controls
            $view = new Zend_View(array('scriptPath' => __DIR__ . '/views/'));
            $pager = $view->paginationControl($news, 'Sliding', 'pagination.phtml', array(
                'listName' => $this->_listName,
                'tags' => urlencode(is_array($this->_tags) ? implode(',', $this->_tags) : $this->_tags)
            ));
        } else {
            if (null === $this->_limit) {
                $this->_limit = Newslog::NEWS_DEFAULT_LIMIT;
            }

            if(!empty($this->_news)) {
                $news = $this->_news;
            } else {
                $news = $this->_mapper->fetchAll($where, $order, $this->_limit, 0, $this->_tags, null, false, $excludedNewsIds);
            }
        }

        if (empty($news)) {
            if (Tools_Security_Acl::isAllowed(Tools_Security_Acl::RESOURCE_PLUGINS)) {
                return $this->_translator->translate('You don\'t have news yet');
            } else {
                return '';
            }
        }

        // Calculation template content once and init prepared list content
        $listContent     = '';

        // prepare correct news list content
        foreach($news as $item) {
            $listContent .= preg_replace('~{\$' . self::NEWS_WIDGET_NAME . ':(.+)}~uU', '{$' . self::NEWS_WIDGET_NAME . ':' . $item->getPageId() . ':$1}', $templateContent);
        }

        //get static containers
        $options = array();

        $containers = $this->_getContainers();
        if(!empty($containers)){
            $options['containers'] = $containers;
        }

        $websiteHelper = Zend_Controller_Action_HelperBroker::getExistingHelper('website');
        $themeData = Zend_Registry::get('theme');
        $extConfig = Zend_Registry::get('extConfig');
        $parserOptions = array(
            'websiteUrl' => $websiteHelper->getUrl(),
            'websitePath' => $websiteHelper->getPath(),
            'currentTheme' => $extConfig['currentTheme'],
            'themePath' => $themeData['path']
        );

        // parse news list content via parser (native widget will be used)
        $parser      = new Tools_Content_Parser($listContent, $options, $parserOptions);
        $listContent = $parser->parse();

        if (!$this->_pagination && $this->_showmore) {
            if ($this->_cleanListOnly) {
                return $listContent;
            }

            $this->_view->newsTemplate = $templateName;

            if (null === $this->_limit) {
                $this->_limit = Newslog::NEWS_DEFAULT_LIMIT;
            }
            $this->_view->limit = $this->_limit;

            if(!empty($this->_tags) && is_array($this->_tags)) {
                $this->_view->tags = implode(',', $this->_tags);
            }

            if(!empty($order) && is_array($order)) {
                $this->_view->sort = implode(',', $order);
            }

            $this->_view->listContent = ($this->_pagination && isset($view)) ? ($listContent . $pager) : $listContent;

            if(!empty($excludedNewsIds) && is_array($excludedNewsIds)) {
                $this->_view->excludedNewsIds = implode(',', $excludedNewsIds);
            }

            return $this->_view->render('newslist.phtml');
        } else {
            return ($this->_pagination && isset($view)) ? ($listContent . $pager) : $listContent;
        }
    }

    /**
     * @return array
     * static containers
     */
    protected function _getContainers(){
        $cDbTable = new Application_Model_DbTable_Container();
        $select = $cDbTable->getAdapter()->select()->from('container', array(
            'uniqHash' => new Zend_Db_Expr("MD5(CONCAT_WS('-',`name`, COALESCE(`page_id`, 0), `container_type`))"),
            'id',
            'name',
            'page_id',
            'container_type',
            'content',
            'published',
            'publishing_date'
        ))
            ->where('(container_type = '.Application_Model_Models_Container::TYPE_STATICCONTENT.' OR container_type = '.Application_Model_Models_Container::TYPE_STATICHEADER.')')
            ->orWhere('page_id IS NULL');

        return $cDbTable->getAdapter()->fetchAssoc($select);
    }

    private function _parseOptions() {
        $excludeKeys = array();
        if(is_array($this->_options) && !empty($this->_options)) {
            foreach($this->_options as $key => $option) {
                //options for order news
                if (array_key_exists($key, $excludeKeys)) {
                    continue;
                }

                if(in_array($option, $this->_allowedOrderParams)){
                    $sortKey = array_search($option, $this->_options);
                    unset($this->_options[$sortKey]);

                    $this->_orderByCol = $this->_allowedOrderColumnsNames[$option];
                    continue;
                }
                // if option is an integer value we assume 'limit' (per_page) passed
                $limit = intval($option);
                if($limit) {
                    $this->_limit = $limit;
                    continue;
                }

                // if options is one of the order direction values we assume new order direction passed
                if($option == self::LOD_ASC || $option == self::LOD_DESC) {
                    $this->_orderDirection = $option;
                    continue;
                }
                // if option is 'tags' check if tags are passed.
                if($option == self::OPT_TAGS) {
                    $tagsSpot    = $key + 1;
                    $this->_tags = !empty($this->_options[$tagsSpot]) ? explode(',', $this->_options[$tagsSpot]) : array();
                    unset($this->_options[$tagsSpot]);
                    $excludeKeys[$tagsSpot] = $tagsSpot;
                    continue;
                }
                // if option is pgr we assume that pagination is needed
                if($option == self::OPT_PAGER) {
                    $this->_pagination = true;
                    $this->_limit      = $this->_limit ? $this->_limit : Newslog::NEWS_DEFAULT_LIMIT;
                    continue;
                }

                if($option === self::NEWS_EXCLUDE_EVENTS){
                    $this->_excludeEvents = true;
                    continue;
                }

                if (strpos($option, self::OPT_DATE) !== false) {
                    continue;
                }

                // if option is list tag, skip it
                //if(in_array($option, $this->_tags)) {continue;}
                // if option didn't go anywhere assume this is a list name
                $this->_listName = $option;
                continue;
            }
        }
    }

    /**
     * @param $cleanListOnly
     * @return $this
     */
    public function setCleanListOnly($cleanListOnly) {
        $this->_cleanListOnly = $cleanListOnly;
        return $this;
    }

    /**
     * @param $news
     * @return $this
     */
    public function setNews($news) {
        $this->_news = $news;
        return $this;
    }
}
